<?php
// check_status.php
// This script checks the session status instantly without reloading the main page.

// Secure session cookie settings and start the session
session_set_cookie_params([
    'httponly' => true,
    'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443),
    'samesite' => 'Lax'
]);
session_start();

// NOTE: Adjust the paths below if your config/ and includes/ folders are located differently
require_once '../config/db.php';
require_once '../includes/utils.php';

// Set the header to JSON
header('Content-Type: application/json');

// We need the fingerprint to validate against the database record
$userFingerprint = generateFingerprint();

if (isset($_SESSION['is_logged_in'])) {
    // Run the core validation. The function updates the last_activity if active, 
    // and UNSETS session variables and returns false if the user was kicked out.
    if (validateActiveSession($pdo, $userFingerprint)) {
        // Validation success (session still active)
        echo json_encode(['status' => 'active']);
    } else {
        // Validation failed (kicked out by a new login, session vars are cleared)
        $reason = 'expired';
        try {
            // Determine if there is an active session for the same code but different session_id
            if (isset($_SESSION['current_code'])) {
                $ttl = getenv('ACTIVE_SESSION_TTL_MINUTES');
                $ttl = ($ttl && is_numeric($ttl) && (int)$ttl > 0) ? (int)$ttl : 30;
                $stmt = $pdo->prepare("SELECT session_id FROM active_sessions WHERE code = ? AND last_activity > (NOW() - INTERVAL ? MINUTE) LIMIT 1");
                $stmt->execute([$_SESSION['current_code'], $ttl]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($row && (!isset($_SESSION['id']) || $row['session_id'] !== session_id())) {
                    $reason = 'taken_over';
                }
            }
        } catch (PDOException $e) {
            // default to expired when uncertain
        }
        session_destroy();
        echo json_encode(['status' => 'kicked_out', 'reason' => $reason]);
    }
} else {
    // No logged-in session variables found
    echo json_encode(['status' => 'logged_out']);
}
?>