<?php
// Admin Password Reset Handler
require_once __DIR__ . '/../config/db.php';

$message = '';
$messageType = '';
$validToken = false;
$token = $_GET['token'] ?? '';

if ($token) {
    // Verify token
    try {
        $stmt = $pdo->prepare("SELECT id, email, expires_at, used FROM admin_reset_tokens WHERE token = ? LIMIT 1");
        $stmt->execute([$token]);
        $tokenData = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$tokenData) {
            $message = 'Invalid reset link';
            $messageType = 'error';
        } elseif ($tokenData['used']) {
            $message = 'This reset link has already been used';
            $messageType = 'error';
        } elseif (strtotime($tokenData['expires_at']) < time()) {
            $message = 'This reset link has expired';
            $messageType = 'error';
        } else {
            $validToken = true;
        }
    } catch (PDOException $e) {
        $message = 'Error verifying token';
        $messageType = 'error';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $validToken) {
    $newCode = trim($_POST['new_code'] ?? '');
    $confirmCode = trim($_POST['confirm_code'] ?? '');
    
    if ($newCode === '' || $newCode !== $confirmCode) {
        $message = 'Codes do not match or are empty';
        $messageType = 'error';
    } elseif (strlen($newCode) < 6) {
        $message = 'Admin code must be at least 6 characters';
        $messageType = 'error';
    } else {
        try {
            $pdo->beginTransaction();
            
            // Update admin code
            $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES ('admin_code', ?)");
            $stmt->execute([$newCode]);
            
            // Mark token as used
            $stmt = $pdo->prepare("UPDATE admin_reset_tokens SET used = 1 WHERE token = ?");
            $stmt->execute([$token]);
            
            $pdo->commit();
            
            $message = 'Admin code successfully reset! You can now log in with your new code.';
            $messageType = 'success';
            $validToken = false; // Prevent form from showing again
        } catch (PDOException $e) {
            $pdo->rollBack();
            $message = 'Error resetting code: ' . $e->getMessage();
            $messageType = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Admin Code - PPV System</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-gray-200 min-h-screen flex items-center justify-center">
    <div class="max-w-md w-full px-4">
        <div class="bg-gray-800 p-8 rounded-xl border border-purple-500/50 shadow-2xl">
            <h1 class="text-3xl font-bold text-white mb-2">Reset Admin Code</h1>
            <p class="text-gray-400 mb-6">Enter your new admin code</p>
            
            <?php if ($message): ?>
            <div class="mb-6 p-4 rounded-lg <?= $messageType === 'error' ? 'bg-red-900/50 border border-red-600 text-red-200' : 'bg-green-900/50 border border-green-600 text-green-200' ?>">
                <?= htmlspecialchars($message) ?>
            </div>
            <?php endif; ?>
            
            <?php if ($validToken): ?>
            <form method="POST" class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-2">New Admin Code</label>
                    <input type="text" name="new_code" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500" minlength="6" maxlength="64" required>
                    <p class="text-xs text-gray-400 mt-1">Minimum 6 characters</p>
                </div>
                <div>
                    <label class="block text-sm font-medium mb-2">Confirm Admin Code</label>
                    <input type="text" name="confirm_code" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500" minlength="6" maxlength="64" required>
                </div>
                <button type="submit" class="w-full py-3 bg-purple-600 hover:bg-purple-700 text-white font-semibold rounded-lg transition">
                    Reset Admin Code
                </button>
            </form>
            <?php endif; ?>
            
            <div class="mt-6 pt-6 border-t border-gray-700 text-center">
                <a href="index.php" class="text-purple-400 hover:text-purple-300 text-sm">← Back to Login</a>
            </div>
        </div>
    </div>
</body>
</html>
