/**
 * Custom JavaScript for controlling the YouTube Iframe API player.
 * This is designed to enforce a custom user experience and restrict direct interaction
 * with the native player controls (as intended by the PHP application structure).
 */

let player;
let interval; // legacy; no longer used for controls
let rafId;
let statusInterval;
let pendingUserPlay = false;
let autoplayFallbackAttached = false;
let autoplayFallbackListener = null;
let progressIntervalId;
const playerContainer = document.getElementById('youtube-player');
const controlsDiv = document.getElementById('custom-controls');
const loadingOverlay = document.getElementById('loading-overlay');
const playPauseButton = document.getElementById('play-pause');
const seekbar = document.getElementById('seek-bar');
const currentTimeSpan = document.getElementById('current-time');
const durationSpan = document.getElementById('duration');
const muteToggleButton = document.getElementById('mute-toggle');
const volumeBar = document.getElementById('volume-bar');
const fullscreenToggle = document.getElementById('fullscreen-toggle');
const reloadVideoButton = document.getElementById('reload-video');
const autoplayHint = document.getElementById('autoplay-hint');

// SVG icons for player controls
const iconPlay = '<svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
const iconPause = '<svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 9v6m4-6v6m7-3a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
const iconVolume = '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.536 8.464a5 5 0 010 7.072m2.828-9.9a9 9 0 010 12.728M5.586 15H4a1 1 0 01-1-1v-4a1 1 0 011-1h1.586l4.707-4.707C10.923 3.663 12 4.109 12 5v14c0 .891-1.077 1.337-1.707.707L5.586 15z"></path></svg>';
const iconMuted = '<svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.536 8.464a5 5 0 010 7.072m2.828-9.9a9 9 0 010 12.728M5.586 15H4a1 1 0 01-1-1v-4a1 1 0 011-1h1.586l4.707-4.707C10.923 3.663 12 4.109 12 5v14c0 .891-1.077 1.337-1.707.707L5.586 15zM19 12h-4"></path></svg>';


/**
 * Converts seconds into a formatted time string (MM:SS).
 * @param {number} seconds
 * @returns {string} Formatted time.
 */
function formatTime(seconds) {
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = Math.floor(seconds % 60);
    const paddedMinutes = String(minutes).padStart(2, '0');
    const paddedSeconds = String(remainingSeconds).padStart(2, '0');
    return `${paddedMinutes}:${paddedSeconds}`;
}

function onPlayerError(event) {
    // event.data codes: 2 (invalid param), 5 (HTML5), 101/150 (embedding disabled)
    try {
        if (loadingOverlay) loadingOverlay.style.display = 'none';
        if (controlsDiv) controlsDiv.classList.add('hidden');
        const container = document.querySelector('.player-container');
        if (container) {
            const msg = document.createElement('div');
            msg.className = 'absolute inset-0 flex items-center justify-center bg-black/70 text-white text-sm p-4';
            let reason = 'Playback error. Please try again later.';
            if (event && typeof event.data === 'number') {
                if (event.data === 2) reason = 'Invalid video ID provided.';
                else if (event.data === 5) reason = 'Playback not supported by your browser.';
                else if (event.data === 101 || event.data === 150) reason = 'The video owner has disabled embedding.';
            }
            const ytLink = (typeof window.videoId === 'string' && window.videoId.length === 11)
              ? `https://www.youtube.com/watch?v=${window.videoId}`
              : null;
            msg.innerHTML = `<div class="text-center max-w-md">
                <p class="mb-3">${reason}</p>
                ${ytLink ? `<a href="${ytLink}" target="_blank" rel="noopener" class="inline-block px-3 py-2 bg-purple-600 hover:bg-purple-700 rounded">Open on YouTube</a>` : ''}
            </div>`;
            container.appendChild(msg);
        }
    } catch (_) {}
}

/**
 * Updates the custom controls (time, seekbar, play/pause icon).
 */
function updateControls() {
    if (player && typeof player.getCurrentTime === 'function') {
        const currentTime = Number(player.getCurrentTime()) || 0;
        const duration = Number(player.getDuration()) || 0;

        // Update Time Display
        currentTimeSpan.textContent = formatTime(currentTime);
        durationSpan.textContent = formatTime(duration);

        // Update Seek Bar (max value needs duration)
        if (isFinite(duration) && duration > 0) {
            seekbar.max = duration;
            seekbar.value = Math.min(currentTime, duration);
        }

        // Update Play/Pause Button
        const playerState = player.getPlayerState();
        if (playerState === YT.PlayerState.PLAYING) {
            playPauseButton.innerHTML = iconPause;
        } else {
            playPauseButton.innerHTML = iconPlay;
        }
    }
}

/**
 * Initializes the YouTube Player.
 * @param {string} videoId The YouTube video ID.
 */
function initPlayer(videoId) {
    if (playerContainer) {
        player = new YT.Player('youtube-player', {
            videoId: videoId,
            playerVars: {
                controls: 0, // Disable native controls
                disablekb: 1, // Disable keyboard controls
                rel: 0, // Disable related videos
                modestbranding: 1, // Minimize YouTube branding
                autoplay: 1, // Attempt autoplay (will succeed when muted)
                fs: 1, // enable fullscreen support
                playsinline: 1 // better iOS behavior
            },
            events: {
                'onReady': onPlayerReady,
                'onStateChange': onPlayerStateChange,
                'onError': onPlayerError
            }
        });
    }
}

/**
 * Fired when the player is ready.
 */
function onPlayerReady(event) {
    loadingOverlay.style.display = 'none';
    controlsDiv.classList.remove('hidden');

    // Initial setup
    // Attempt autoplay without muting (may be blocked by browser policies)
    try { player.playVideo(); } catch (_) {}
    player.setVolume(100);
    // Initialize icons to a consistent state
    try {
        if (player.isMuted && player.isMuted()) {
            muteToggleButton.innerHTML = iconMuted;
        } else {
            muteToggleButton.innerHTML = iconVolume;
        }
    } catch (_) {}
    playPauseButton.innerHTML = iconPlay;
    updateControls();
    
    // Set up control listeners
    setupEventListeners();
    
    // Start continuous update loop via requestAnimationFrame
    startUpdateLoop();

    // If a user clicked play before ready, honor it now
    if (pendingUserPlay) {
        try { player.playVideo(); } catch (_) {}
        pendingUserPlay = false;
    }

    // Start polling session status to detect takeover from another device
    startSessionPolling();

    // Ensure iframe has fullscreen permissions (especially important for iOS Safari)
    try {
        const iframe = typeof player.getIframe === 'function' ? player.getIframe() : document.querySelector('#youtube-player iframe');
        if (iframe) {
            iframe.setAttribute('allow', 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; fullscreen');
            iframe.setAttribute('allowfullscreen', '');
            iframe.setAttribute('webkitallowfullscreen', '');
            iframe.setAttribute('mozallowfullscreen', '');
        }
    } catch (_) {}

    setTimeout(() => {
        try {
            const st = player.getPlayerState ? player.getPlayerState() : null;
            if (st !== YT.PlayerState.PLAYING) {
                if (!autoplayFallbackAttached) {
                    autoplayFallbackAttached = true;
                    if (autoplayHint) autoplayHint.setAttribute('data-show', 'true');
                    autoplayFallbackListener = () => {
                        try {
                            player.playVideo();
                            if (player.isMuted && player.isMuted()) {
                                player.unMute();
                                muteToggleButton.innerHTML = iconVolume;
                            }
                        } catch (_) {}
                        if (autoplayFallbackAttached) {
                            window.removeEventListener('pointerdown', autoplayFallbackListener, { once: true });
                            window.removeEventListener('keydown', autoplayFallbackListener, { once: true });
                            autoplayFallbackAttached = false;
                            autoplayFallbackListener = null;
                        }
                        if (autoplayHint) autoplayHint.setAttribute('data-show', 'false');
                    };
                    window.addEventListener('pointerdown', autoplayFallbackListener, { once: true });
                    window.addEventListener('keydown', autoplayFallbackListener, { once: true });
                }
            }
        } catch (_) {}
    }, 800);

    // Load saved viewing position (Continue Watching)
    try {
        if (typeof window.videoId === 'string' && window.videoId.length) {
            fetch(`progress.php?video=${encodeURIComponent(window.videoId)}`, { credentials: 'same-origin' })
                .then(r => r.ok ? r.json() : { ok: false })
                .then(data => {
                    if (data && data.ok && typeof data.position === 'number' && data.position > 0) {
                        try { player.seekTo(data.position, true); } catch (_) {}
                    }
                })
                .catch(() => {});
        }
    } catch (_) {}

    // Start periodic progress saving every 15s
    try { if (progressIntervalId) clearInterval(progressIntervalId); } catch (_) {}
    progressIntervalId = setInterval(() => {
        saveProgress();
        trackAnalytics();
    }, 15000);
    
    // Track video start
    trackVideoStart();
}

function trackVideoStart() {
    try {
        if (!window.videoId) return;
        fetch('analytics.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify({ action: 'start', video: window.videoId })
        }).catch(() => {});
    } catch (_) {}
}

function trackAnalytics() {
    try {
        if (!player || typeof player.getCurrentTime !== 'function') return;
        if (!window.videoId) return;
        const pos = Math.floor(Number(player.getCurrentTime()) || 0);
        const dur = Math.floor(Number(player.getDuration()) || 0);
        if (pos < 0 || dur <= 0) return;
        fetch('analytics.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify({ action: 'update', video: window.videoId, position: pos, duration: dur })
        }).catch(() => {});
    } catch (_) {}
}

/**
 * Fired when the player's state changes (Playing, Paused, Ended, etc.).
 */
function onPlayerStateChange(event) {
    updateControls();
    try {
        if (event && event.data === YT.PlayerState.PLAYING && autoplayFallbackAttached && autoplayFallbackListener) {
            window.removeEventListener('pointerdown', autoplayFallbackListener, { once: true });
            window.removeEventListener('keydown', autoplayFallbackListener, { once: true });
            autoplayFallbackAttached = false;
            autoplayFallbackListener = null;
        }
        if (event && event.data === YT.PlayerState.PLAYING) {
            if (autoplayHint) autoplayHint.setAttribute('data-show', 'false');
        }
    } catch (_) {}
}

function startUpdateLoop() {
    if (rafId) cancelAnimationFrame(rafId);
    const tick = () => {
        updateControls();
        rafId = requestAnimationFrame(tick);
    };
    rafId = requestAnimationFrame(tick);
}

// Cleanup on navigate away
window.addEventListener('beforeunload', () => {
    try { if (rafId) cancelAnimationFrame(rafId); } catch (_) {}
    try { if (interval) clearInterval(interval); } catch (_) {}
    try { if (progressIntervalId) clearInterval(progressIntervalId); } catch (_) {}
    try {
        if (autoplayFallbackAttached && autoplayFallbackListener) {
            window.removeEventListener('pointerdown', autoplayFallbackListener, { once: true });
            window.removeEventListener('keydown', autoplayFallbackListener, { once: true });
        }
    } catch (_) {}
    try { saveProgress(true); } catch (_) {}
});

function saveProgress(useBeacon = false) {
    try {
        if (!player || typeof player.getCurrentTime !== 'function') return;
        if (!window.videoId) return;
        const pos = Math.floor(Number(player.getCurrentTime()) || 0);
        if (pos < 0) return;
        const payload = JSON.stringify({ video: window.videoId, position: pos });
        if (useBeacon && 'navigator' in window && typeof navigator.sendBeacon === 'function') {
            const blob = new Blob([payload], { type: 'application/json' });
            navigator.sendBeacon('progress.php', blob);
        } else {
            fetch('progress.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'same-origin',
                body: payload
            }).catch(() => {});
        }
    } catch (_) {}
}

/**
 * Sets up listeners for the custom control elements.
 */
function setupEventListeners() {
    // Play/Pause Toggle
    playPauseButton.addEventListener('click', () => {
        if (!player || typeof player.getPlayerState !== 'function') {
            // Player not ready yet; queue a play
            pendingUserPlay = true;
            return;
        }
        const playerState = player.getPlayerState();
        try {
            if (playerState === YT.PlayerState.PLAYING) {
                player.pauseVideo();
            } else {
                // Unmute on explicit user play if volume is above 0
                try {
                    const vol = Number(player.getVolume()) || 100;
                    if (vol > 0 && player.isMuted && player.isMuted()) {
                        player.unMute();
                        muteToggleButton.innerHTML = iconVolume;
                    }
                } catch (_) {}
                player.playVideo();
            }
        } catch (_) {}
    });

    // Seek Bar (Scrubbing)
    seekbar.addEventListener('input', () => {
        if (!player) return;
        const duration = Number(player.getDuration()) || 0;
        if (!(isFinite(duration) && duration > 0)) return;
        try {
            player.pauseVideo();
            player.seekTo(Number(seekbar.value) || 0, false);
        } catch (_) {}
    });
    seekbar.addEventListener('change', () => {
        if (!player) return;
        const duration = Number(player.getDuration()) || 0;
        if (!(isFinite(duration) && duration > 0)) return;
        try {
            player.seekTo(Number(seekbar.value) || 0, true);
            player.playVideo();
        } catch (_) {}
    });

    // Mute Toggle
    muteToggleButton.addEventListener('click', () => {
        if (player.isMuted()) {
            player.unMute();
            muteToggleButton.innerHTML = iconVolume;
            volumeBar.value = player.getVolume();
        } else {
            player.mute();
            muteToggleButton.innerHTML = iconMuted;
            volumeBar.value = 0;
        }
    });
    
    // Volume Control
    volumeBar.addEventListener('input', () => {
        player.setVolume(volumeBar.value);
        if (player.isMuted() && volumeBar.value > 0) {
            player.unMute();
            muteToggleButton.innerHTML = iconVolume;
        } else if (volumeBar.value == 0 && !player.isMuted()) {
            player.mute();
            muteToggleButton.innerHTML = iconMuted;
        } else if (volumeBar.value > 0 && player.isMuted()) {
             player.unMute();
            muteToggleButton.innerHTML = iconVolume;
        }
    });

    // Fullscreen Toggle
    fullscreenToggle.addEventListener('click', () => {
        const videoWrapper = document.querySelector('.player-container');
        if (!videoWrapper) return;
        const doc = document;
        const inFs = doc.fullscreenElement || doc.webkitFullscreenElement || doc.mozFullScreenElement || doc.msFullscreenElement;
        const exit = doc.exitFullscreen || doc.webkitExitFullscreen || doc.mozCancelFullScreen || doc.msExitFullscreen;
        if (!inFs) {
            // Try wrapper first
            const reqWrapper = videoWrapper.requestFullscreen || videoWrapper.webkitRequestFullscreen || videoWrapper.mozRequestFullScreen || videoWrapper.msRequestFullscreen;
            let requested = false;
            try {
                if (reqWrapper) { reqWrapper.call(videoWrapper); requested = true; }
            } catch (_) {}
            // iOS Safari fallback: try the iframe directly
            if (!requested) {
                try {
                    const iframe = typeof player.getIframe === 'function' ? player.getIframe() : document.querySelector('#youtube-player iframe');
                    if (iframe) {
                        const reqIframe = iframe.requestFullscreen || iframe.webkitRequestFullscreen;
                        if (reqIframe) { reqIframe.call(iframe); requested = true; }
                    }
                } catch (_) {}
            }
            if (!requested) {
                // Last resort
                try { doc.documentElement.requestFullscreen && doc.documentElement.requestFullscreen(); } catch (_) {}
            }
        } else {
            try { exit.call(doc); } catch (err) { console.error('Exit fullscreen error:', err); }
        }
    });

    // Reload Video Button
    reloadVideoButton.addEventListener('click', () => {
        // Stops the player loop
        clearInterval(interval);
        
        // Show loading overlay
        loadingOverlay.style.display = 'flex';
        controlsDiv.classList.add('hidden');
        
        // Reload the page to fetch the potentially new video ID from the server
        window.location.reload();
    });
}

/**
 * Poll the server periodically to ensure this session is still valid.
 * If the session was taken over by another device, stop playback and redirect.
 */
function startSessionPolling() {
    // Safety: clear existing interval if any
    if (statusInterval) clearInterval(statusInterval);

    statusInterval = setInterval(async () => {
        try {
            const res = await fetch('status_check.php', { cache: 'no-store', credentials: 'same-origin' });
            if (!res.ok) return; // ignore transient errors
            const data = await res.json();
            if (data.status === 'kicked_out' || data.status === 'logged_out') {
                // Stop timers and pause video immediately
                if (interval) clearInterval(interval);
                if (player && player.pauseVideo) {
                    try { player.pauseVideo(); } catch (_) {}
                }
                if (statusInterval) clearInterval(statusInterval);
                // Redirect without hitting the logout endpoint (to avoid nuking the new device session)
                window.location.replace('index.php');
            }
        } catch (e) {
            // Network error; ignore and retry on next tick
        }
    }, 5000); // check every 5 seconds
}
