<?php
// Set display errors to help with debugging (should be off in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Basic security headers (adjust CSP as needed for your deployment)
header("Referrer-Policy: no-referrer");
header("X-Frame-Options: SAMEORIGIN");
header("X-Content-Type-Options: nosniff");
header("Permissions-Policy: fullscreen=(self)");
header("Content-Security-Policy: default-src 'self'; img-src 'self' data: https:; script-src 'self' 'unsafe-inline' https://www.youtube.com https://cdn.tailwindcss.com; style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; frame-src https://www.youtube.com https://www.youtube-nocookie.com; connect-src 'self'; font-src 'self' https://fonts.gstatic.com;");

// Secure session cookie settings (before session_start)
session_set_cookie_params([
    'httponly' => true,
    'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443),
    'samesite' => 'Lax'
]);
session_start();
// Robust includes using absolute paths
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/utils.php';

// Generate a unique fingerprint for the current user/device
$userFingerprint = generateFingerprint();
$defaultVideoId = '_0vEP79i9bQ'; // Default fallback video (e.g., a test video)
$adminCode = 'ADMIN123';
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'admin_code' LIMIT 1");
    $stmt->execute();
    $dbAdmin = $stmt->fetchColumn();
    if ($dbAdmin !== false && $dbAdmin !== null && $dbAdmin !== '') { $adminCode = $dbAdmin; }
} catch (PDOException $e) { /* ignore and fallback */ }
if ($adminCode === 'ADMIN123') {
    $envAdmin = getenv('ADMIN_CODE');
    if ($envAdmin) { $adminCode = $envAdmin; }
}

// --- Session Validation ---
// Skip validation on the very first load after login to avoid edge-case timing issues
if (isset($_SESSION['just_logged_in'])) {
    unset($_SESSION['just_logged_in']);
} elseif (isset($_SESSION['is_logged_in']) && !isset($_SESSION['is_admin']) && !validateActiveSession($pdo, $userFingerprint)) {
    // Session invalidated - clear auth flags but keep session alive to show message
    unset($_SESSION['is_logged_in'], $_SESSION['is_admin'], $_SESSION['current_code']);
    showMessage('Your session was terminated or taken over by another device.', true);
    // Ensure session data is saved before redirect
    session_write_close();
    // Ensure session data is saved before redirect
    session_write_close();
    header('Location: index.php');
    exit;
}

// --- Load Active Video (default) from Settings ---
$videoUrl = $defaultVideoId; // Fallback
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'active_video_id'");
    $stmt->execute();
    $result = $stmt->fetchColumn();
    if ($result !== false) {
        $videoUrl = $result;
    }
} catch (PDOException $e) {
    error_log("Error accessing settings: " . $e->getMessage());
    showMessage("Error accessing settings: " . $e->getMessage(), true);
}

// --- Compute list of available videos for this code (entitlements + schedule windows) ---
$availableVideos = [];
if (isset($_SESSION['is_logged_in']) && !isset($_SESSION['is_admin'])) {
    try {
        // Entitled videos first
        $stmt = $pdo->prepare("SELECT v.id, v.youtube_id, v.title, v.artwork_url, v.synopsis
                               FROM code_entitlements ce JOIN videos v ON v.id = ce.video_id
                               WHERE ce.code = ?");
        $stmt->execute([$_SESSION['current_code']]);
        $entitled = $stmt->fetchAll(PDO::FETCH_ASSOC);
        // Filter by schedules: if a video has schedules, it must be within at least one active window; if no schedules, it's available any time
        foreach ($entitled as $row) {
            $vidId = (int)$row['id'];
            $hasSchedules = false; $within = false;
            $stmt2 = $pdo->prepare("SELECT starts_at, ends_at FROM schedules WHERE video_id = ?");
            $stmt2->execute([$vidId]);
            $schRows = $stmt2->fetchAll(PDO::FETCH_ASSOC);
            if ($schRows && count($schRows) > 0) {
                $hasSchedules = true;
                $stmt3 = $pdo->prepare("SELECT 1 FROM schedules WHERE video_id = ? AND NOW() BETWEEN starts_at AND ends_at LIMIT 1");
                $stmt3->execute([$vidId]);
                $within = (bool)$stmt3->fetchColumn();
            }
            // Compute next upcoming start if not currently within window
            $nextStart = null; $nextEnd = null;
            if ($hasSchedules && !$within) {
                $stmt4 = $pdo->prepare("SELECT starts_at, ends_at FROM schedules WHERE video_id = ? AND starts_at > NOW() ORDER BY starts_at ASC LIMIT 1");
                $stmt4->execute([$vidId]);
                $next = $stmt4->fetch(PDO::FETCH_ASSOC);
                if ($next) { $nextStart = $next['starts_at']; $nextEnd = $next['ends_at']; }
            }
            // Always list entitled videos in the grid; enforce availability at playback level already handled below
            $row['is_live'] = (!$hasSchedules || $within) ? 1 : 0;
            $row['next_start'] = $nextStart;
            $row['next_end'] = $nextEnd;
            $availableVideos[] = $row; // id, youtube_id, title, artwork_url, synopsis, is_live, next_start
        }
    } catch (PDOException $e) {
        error_log('Available videos compute error: ' . $e->getMessage());
    }
}

// Selected video resolution: regular users always use admin active video; only admins can override via ?v
$selectedYoutube = isset($_GET['v']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', $_GET['v']) : '';
if (isset($_SESSION['is_admin'])) {
    if ($selectedYoutube) { $videoUrl = $selectedYoutube; }
} else if (isset($_SESSION['is_logged_in'])) {
    // For non-admin logged-in users, force admin-configured active video; do not switch to others
    // $videoUrl is already loaded from settings above; leave as-is
    // If they have no entitlement/schedule for it, playback will be blocked below
} else {
    // Not logged in: keep admin-configured video for hero/play preview
}

// --- Entitlement & Scheduling Enforcement ---
$allowPlayback = true;
$blockedReason = '';
if (isset($_SESSION['is_logged_in']) && !isset($_SESSION['is_admin'])) {
    try {
        // Find video row if exists
        $stmt = $pdo->prepare("SELECT id FROM videos WHERE youtube_id = ? LIMIT 1");
        $stmt->execute([$videoUrl]);
        $videoRow = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($videoRow && isset($videoRow['id'])) {
            $vidId = (int)$videoRow['id'];
            // Check entitlement: code -> video
            $stmt = $pdo->prepare("SELECT 1 FROM code_entitlements WHERE code = ? AND video_id = ? LIMIT 1");
            $stmt->execute([$_SESSION['current_code'], $vidId]);
            if (!$stmt->fetchColumn()) {
                $allowPlayback = false;
                $blockedReason = 'not_entitled';
            } else {
                // If schedules exist for this video, enforce current time inside at least one window
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM schedules WHERE video_id = ?");
                $stmt->execute([$vidId]);
                $totalSchedules = (int)$stmt->fetchColumn();
                if ($totalSchedules > 0) {
                    $stmt = $pdo->prepare("SELECT 1 FROM schedules WHERE video_id = ? AND NOW() BETWEEN starts_at AND ends_at LIMIT 1");
                    $stmt->execute([$vidId]);
                    if (!$stmt->fetchColumn()) {
                        $allowPlayback = false;
                        $blockedReason = 'out_of_window';
                    }
                }
            }
        }
    } catch (PDOException $e) {
        error_log('Entitlement/Schedule check error: ' . $e->getMessage());
        // Fail closed: block playback if DB check fails
        $allowPlayback = false;
        $blockedReason = 'validation_error';
    }
}

// --- Handle Login ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['access_code'])) {
    $codeInput = strtoupper(trim($_POST['access_code']));
    
    // Check for Admin Code
    if ($codeInput === $adminCode) {
        $_SESSION['is_admin'] = true;
        $_SESSION['is_logged_in'] = true;
        $_SESSION['current_code'] = $adminCode;
        // Regenerate session ID to finalize login and prevent fixation
        session_regenerate_id(true);
        // Do NOT record admin in active_sessions (FK requires access_codes entry). Admin is exempt from single-device enforcement.
        // mark to skip validation on the next page load
        $_SESSION['just_logged_in'] = true;
        if (function_exists('logAudit')) { logAudit($pdo, 'login_success', 'admin'); }
        showMessage('Admin access granted.', false);
        header('Location: admin.php');
        exit;
    } else {
        // Check database for a valid access code
        try {
            $stmt = $pdo->prepare("SELECT * FROM access_codes WHERE code = ?");
            $stmt->execute([$codeInput]);
            $codeRecord = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($codeRecord) {
                // Code found: log user in, update the access_codes table
                $updateStmt = $pdo->prepare("UPDATE access_codes SET used_by = ?, date_used = NOW() WHERE code = ?");
                $updateStmt->execute([$userFingerprint, $codeInput]);
                
                $_SESSION['is_logged_in'] = true;
                $_SESSION['current_code'] = $codeInput;
                // Regenerate session ID to finalize login and prevent fixation
                session_regenerate_id(true);
                // Invalidate old session for this code and create a new active session using the new session id
                invalidateOldSession($pdo, $codeInput, $userFingerprint, session_id());
                // mark to skip validation on the next page load
                $_SESSION['just_logged_in'] = true;
                if (function_exists('logAudit')) { logAudit($pdo, 'login_success', 'code='.$codeInput); }
                showMessage('Access Granted! Your code is now tied to this device.', false);
            } else {
                if (function_exists('logAudit')) { logAudit($pdo, 'login_fail', 'code='.$codeInput); }
                showMessage('Invalid access code.', true);
            }
        } catch (PDOException $e) {
            error_log("Database error during login: " . $e->getMessage());
            showMessage("Database error: " . $e->getMessage(), true);
        }
    }
    header('Location: index.php');
    exit;
}

// --- Handle Logout ---
if (isset($_GET['logout'])) {
    if (isset($_SESSION['current_code'])) {
        try {
            // Delete only the current session record for this code/session
            $stmt = $pdo->prepare("DELETE FROM active_sessions WHERE code = ? AND session_id = ?");
            $stmt->execute([$_SESSION['current_code'], session_id()]);
        } catch (PDOException $e) {
            error_log("Error cleaning up session on logout: " . $e->getMessage());
        }
    }
    if (function_exists('logAudit')) { logAudit($pdo, 'logout'); }
    session_destroy();
    showMessage('You have been logged out.', false);
    header('Location: index.php');
    exit;
}

// --- Load codes for admin view ---
$codes = [];
if (isset($_SESSION['is_admin'])) {
    try {
        $stmt = $pdo->query("SELECT * FROM access_codes ORDER BY date_created DESC");
        $codes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error loading codes for admin: " . $e->getMessage());
        showMessage("Error loading codes: " . $e->getMessage(), true);
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Harmonious Chorale | Heaven 2</title>
    <!-- YouTube Iframe API -->
    <script src="https://www.youtube.com/iframe_api"></script>
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Custom Player JS (Assumed to be in js/player.js relative to the server root) -->
    <script src="js/player.js" defer></script>
    <style>
        /* Custom styles for the range input (seek and volume bars) */
        input[type=range]::-webkit-slider-thumb {
            -webkit-appearance: none;
            appearance: none;
            width: 16px;
            height: 16px;
            border-radius: 50%;
            background: #a855f7; /* purple-500 */
            cursor: pointer;
        }
        input[type=range]::-moz-range-thumb {
            width: 16px;
            height: 16px;
            border-radius: 50%;
            background: #a855f7;
            cursor: pointer;
        }
        /* Volume Bar: make it thinner */
        #volume-bar {
            height: 4px;
        }

        /* Ensure YouTube iframe fills the player box */
        #youtube-player {
            position: absolute;
            top: 0; left: 0; right: 0; bottom: 0;
            width: 100%;
            height: 100%;
        }
        #youtube-player iframe {
            width: 100% !important;
            height: 100% !important;
            display: block;
        }
        /* Autoplay hint overlay */
        .autoplay-hint { position: absolute; inset: 0; display: none; align-items: center; justify-content: center; z-index: 25; }
        .autoplay-hint[data-show="true"] { display: flex; }

        /* Page background with dark gradient overlay for readability */
        body {
            background-image: linear-gradient(rgba(0,0,0,0.7), rgba(0,0,0,0.7)), url('img/background.jpg');
            background-size: cover;
            background-position: center center;
            background-repeat: no-repeat;
            background-attachment: fixed;
        }
    </style>
</head>
<body class="min-h-screen bg-gray-950 flex flex-col items-center p-4 sm:p-8 font-sans">
    <!-- WIDER PAGE HEADER: max-w-7xl -->
    <header class="w-full text-center mb-6 max-w-7xl">
        <!--<div class="flex items-center justify-center gap-3">
            <img src="img/chazon.png" alt="Site Logo" class="h-10 w-auto select-none" loading="eager">
            <h1 class="text-4xl font-extrabold text-white tracking-tight">
                PPV Access Manager
            </h1>
        </div>-->
        <?php if (isset($_SESSION['is_logged_in'])): ?>
            <div class="flex justify-center items-center mt-3">
                <?php if (isset($_SESSION['is_admin'])): ?>
                    <p class="text-sm font-mono text-yellow-300 mr-4">ADMIN MODE</p>
                <?php endif; ?>
                <?php if (isset($_SESSION['is_logged_in']) && !isset($_SESSION['is_admin']) && count($availableVideos) > 1): ?>
                    <div class="mb-6 p-4 bg-gray-800 rounded-lg border border-gray-700">
                        <h3 class="text-lg font-semibold text-white mb-3">Choose an event</h3>
                        <div class="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
                            <?php foreach ($availableVideos as $av): ?>
                                <div class="block rounded-lg bg-gray-700 overflow-hidden">
                                    <a href="?v=<?= htmlspecialchars($av['youtube_id']) ?>" class="block">
                                        <?php if (!empty($av['artwork_url'])): ?>
                                            <img src="<?= htmlspecialchars($av['artwork_url']) ?>" alt="<?= htmlspecialchars($av['title']) ?>" class="w-full h-32 object-cover">
                                        <?php endif; ?>
                                    </a>
                                    <div class="p-3 flex items-start justify-between gap-2">
                                        <div>
                                            <a href="?v=<?= htmlspecialchars($av['youtube_id']) ?>" class="text-white font-semibold hover:underline">
                                                <?= htmlspecialchars($av['title']) ?>
                                            </a>
                                            <p class="text-xs text-gray-300 font-mono">ID: <?= htmlspecialchars($av['youtube_id']) ?></p>
                                        </div>
                                        <button class="px-2 py-1 text-xs bg-gray-600 hover:bg-gray-500 text-white rounded details-btn"
                                            data-title="<?= htmlspecialchars($av['title']) ?>"
                                            data-synopsis="<?= htmlspecialchars($av['synopsis'] ?? '') ?>"
                                            data-art="<?= htmlspecialchars($av['artwork_url'] ?? '') ?>"
                                            data-live="<?= !empty($av['is_live']) ? '1' : '0' ?>"
                                            data-nextstart="<?= htmlspecialchars($av['next_start'] ?? '') ?>"
                                            data-nextend="<?= htmlspecialchars($av['next_end'] ?? '') ?>"
                                            data-yt="<?= htmlspecialchars($av['youtube_id']) ?>"
                                            data-tags="<?= htmlspecialchars($av['tags'] ?? '') ?>">
                                            Info
                                        </button>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
                <a href="?logout" class="px-4 py-1 bg-red-600 hover:bg-red-700 text-white rounded-full text-sm font-semibold transition duration-200 flex items-center">
                    <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg> Logout
                </a>
            </div>
        <?php endif; ?>
    </header>

    <!-- Display Error Messages (Wider container) -->
    <?php $error = getMessage('error'); if ($error): ?>
        <div class="w-full max-w-7xl bg-red-900/50 border border-red-500 text-red-300 p-3 rounded-lg mb-4 flex items-center">
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
            <p class="font-semibold"><?php echo htmlspecialchars($error); ?></p>
        </div>
    <?php endif; ?>

    <!-- Display Success Messages (Wider container) -->
    <?php $message = getMessage('message'); if ($message): ?>
        <div class="w-full max-w-7xl bg-green-900/50 border border-green-500 text-green-300 p-3 rounded-lg mb-4">
            <p><?php echo htmlspecialchars($message); ?></p>
        </div>
    <?php endif; ?>

    <!-- MAIN CONTENT CONTAINER (Wider) -->
    <main class="flex flex-col items-center w-full max-w-7xl">
        <?php if (isset($_SESSION['is_logged_in'])): ?>
            <!-- LOGGED-IN VIEW (STREAM & ADMIN PANEL) -->
            <!-- The w-full ensures it fills the max-w-7xl container -->
            <div class="mt-4 p-4 sm:p-6 bg-gray-900 rounded-xl shadow-2xl w-full border border-purple-500/50">
                <?php 
                    $profilesEnabled = false; 
                    try { 
                        $pe = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='profiles_enabled'")->fetchColumn(); 
                        $profilesEnabled = ($pe === '1');
                    } catch (PDOException $e) { $profilesEnabled = false; }
                ?>
                <?php if ($profilesEnabled && !isset($_SESSION['profile_id'])): ?>
                    <div id="profile-picker" class="mb-6 p-4 bg-gray-800 rounded-lg border border-gray-700">
                        <h3 class="text-lg font-semibold text-white mb-2">Who's watching?</h3>
                        <div id="profiles-list" class="flex flex-wrap gap-3 mb-3"></div>
                        <div class="flex gap-2 items-center">
                            <input type="text" id="new-profile-name" class="input-text flex-grow" placeholder="Create new profile (name)" maxlength="50">
                            <button id="create-profile" class="px-3 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded">Create</button>
                        </div>
                        <p id="profile-error" class="text-sm text-red-400 mt-2 hidden"></p>
                    </div>
                    <script>
                        (function(){
                            const list = document.getElementById('profiles-list');
                            const err = document.getElementById('profile-error');
                            const input = document.getElementById('new-profile-name');
                            const createBtn = document.getElementById('create-profile');
                            function showError(msg){ err.textContent = msg; err.classList.remove('hidden'); }
                            function hideError(){ err.classList.add('hidden'); }
                            function load(){
                                fetch('profile.php', { credentials: 'same-origin' })
                                  .then(r=>r.ok?r.json():Promise.reject(r))
                                  .then(d=>{
                                      list.innerHTML='';
                                      if (!d || !d.ok) { return; }
                                      (d.profiles||[]).forEach(p=>{
                                          const btn=document.createElement('button');
                                          btn.className='px-3 py-2 bg-gray-700 hover:bg-gray-600 text-white rounded';
                                          btn.textContent=p.name;
                                          btn.addEventListener('click',()=>{
                                              fetch('profile.php',{
                                                  method:'POST',
                                                  headers:{'Content-Type':'application/json'},
                                                  credentials:'same-origin',
                                                  body: JSON.stringify({ action:'select', profile_id: p.id })
                                              }).then(r=>r.ok?r.json():Promise.reject(r)).then(x=>{ if(x&&x.ok){ location.reload(); } else { showError('Could not select profile'); } }).catch(()=>showError('Network error selecting profile'));
                                          });
                                          list.appendChild(btn);
                                      });
                                  }).catch(()=>{});
                            }
                            createBtn.addEventListener('click',()=>{
                                const name = (input.value||'').trim();
                                if (!name){ showError('Enter a name'); return; }
                                if (name.length>50){ showError('Name too long'); return; }
                                hideError();
                                createBtn.disabled = true; createBtn.classList.add('opacity-60','cursor-not-allowed');
                                fetch('profile.php',{
                                    method:'POST',
                                    headers:{'Content-Type':'application/json'},
                                    credentials:'same-origin',
                                    body: JSON.stringify({ action:'create', name })
                                }).then(r=>r.ok?r.json():Promise.reject(r))
                                  .then(x=>{ if(x&&x.ok){ location.reload(); } else { showError('Could not create profile'); } })
                                  .catch(()=>{ showError('Network error creating profile'); })
                                  .finally(()=>{ createBtn.disabled=false; createBtn.classList.remove('opacity-60','cursor-not-allowed'); });
                            });
                            load();
                        })();
                    </script>
                <?php endif; ?>
                <h2 class="text-2xl font-bold text-white mb-4 flex items-center">
                    <svg class="w-6 h-6 mr-2 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"></path></svg> Premium Stream
                </h2>
                <?php 
                    $heroTitle = '';
                    $heroSynopsis = '';
                    $heroImage = '';
                    try { $heroTitle = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_title'")->fetchColumn() ?: ''; } catch (PDOException $e) {}
                    try { $heroSynopsis = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_synopsis'")->fetchColumn() ?: ''; } catch (PDOException $e) {}
                    try { $heroImage = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_image'")->fetchColumn() ?: ''; } catch (PDOException $e) {}
                    $hasHero = !empty($heroTitle) || !empty($heroSynopsis) || !empty($heroImage);
                ?>
                <?php if ($hasHero): ?>
                    <div class="relative overflow-hidden rounded-lg mb-4 border border-gray-700">
                        <div class="absolute inset-0" style="background-image: linear-gradient(rgba(0,0,0,0.6), rgba(0,0,0,0.6)), url('<?= htmlspecialchars($heroImage) ?>'); background-size: cover; background-position: center;"></div>
                        <div class="relative p-6 sm:p-8">
                            <h3 class="text-2xl sm:text-3xl font-bold text-white mb-2"><?= htmlspecialchars($heroTitle) ?></h3>
                            <p class="text-sm sm:text-base text-gray-200 max-w-3xl mb-4"><?= nl2br(htmlspecialchars($heroSynopsis)) ?></p>
                            <?php if (empty($_SESSION['is_logged_in']) || isset($_SESSION['is_admin']) || !empty($availableVideos)): ?>
                                <a id="hero-play" href="<?= !empty($videoUrl) ? '?v='.htmlspecialchars($videoUrl) : '#' ?>" class="inline-block px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg text-sm transition<?= empty($videoUrl) ? ' opacity-50 cursor-not-allowed' : '' ?>">Play</a>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                <?php endif; ?>
                <?php if ($allowPlayback): ?>
                <!-- Video Player Container (Already w-full, now wider due to parent container) -->
                <div class="player-container bg-black rounded-xl overflow-hidden shadow-2xl relative" style="padding-bottom: 56.25%; height: 0;">
                    <!-- Invisible overlay to prevent right-click and controls manipulation -->
                    <div class="absolute inset-0 cursor-default" style="z-index: 15; background-color: transparent;"></div>
                    <div id="youtube-player" class="absolute inset-0" style="z-index: 11;"></div>
                    <!-- Loading Overlay -->
                    <div id="loading-overlay" class="absolute inset-0 bg-gray-900 flex items-center justify-center text-white" style="z-index: 20;">
                        <div class="text-center p-4">
                            <svg class="w-10 h-10 mx-auto text-purple-400 animate-pulse" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"></path></svg>
                            <p class="mt-3">Loading Custom Player...</p>
                        </div>
                    </div>
                    <!-- Autoplay blocked hint -->
                    <div id="autoplay-hint" class="autoplay-hint">
                        <div class="bg-black/70 text-white px-4 py-2 rounded-lg text-sm">
                            Autoplay is blocked. Click anywhere to start playback.
                        </div>
                    </div>
                </div>

                <!-- Custom Player Controls -->
                <div id="custom-controls" class="mt-4 p-3 bg-gray-800 rounded-lg flex flex-wrap items-center space-x-2 sm:space-x-4 hidden">
                    <button id="play-pause" aria-label="Play or pause video" class="p-3 bg-purple-600 hover:bg-purple-700 text-white rounded-full transition duration-150 transform hover:scale-110 flex-shrink-0"></button>
                    <span id="current-time" class="text-sm font-mono text-white min-w-[50px] flex-shrink-0">00:00</span>
                    <input type="range" id="seek-bar" min="0" max="0" step="0.1" value="0" class="flex-grow h-2 rounded-lg appearance-none cursor-pointer bg-gray-700 mx-2">
                    <span id="duration" class="text-sm font-mono text-gray-400 min-w-[50px] flex-shrink-0">00:00</span>
                    <div class="flex items-center space-x-1 flex-shrink-0 w-24 sm:w-32">
                        <button id="mute-toggle" aria-label="Toggle mute" class="p-2 text-gray-400 hover:text-white transition duration-150">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.536 8.464a5 5 0 010 7.072m2.828-9.9a9 9 0 010 12.728M5.586 15H4a1 1 0 01-1-1v-4a1 1 0 011-1h1.586l4.707-4.707C10.923 3.663 12 4.109 12 5v14c0 .891-1.077 1.337-1.707.707L5.586 15z"></path></svg>
                        </button>
                        <input type="range" id="volume-bar" min="0" max="100" value="100" class="w-full h-1 rounded-lg appearance-none cursor-pointer bg-gray-700">
                    </div>
                    <button id="fullscreen-toggle" aria-label="Toggle fullscreen" class="p-2 text-gray-400 hover:text-white transition duration-150">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4"></path></svg>
                    </button>
                </div>
                
                <!-- Access Info / Reload Button -->
                <div class="mt-6 text-center bg-gray-800 rounded-lg p-4 flex justify-between items-center flex-wrap">
                    <p class="text-gray-300 text-sm">Access confirmed.</p>
                    <p class="text-gray-400 text-xs mt-2 sm:mt-0">Device ID: <span class="font-mono text-xs text-purple-300"><?php echo htmlspecialchars(substr($userFingerprint, 0, 12)); ?>...</span></p>
                    <button id="reload-video" aria-label="Reload video" class="px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg text-sm transition duration-200 mt-2 sm:mt-0">
                        <svg class="w-4 h-4 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg> Reload Video
                    </button>
                </div>
                <?php else: ?>
                <div class="p-4 bg-gray-800 rounded-lg border border-red-600/50 text-red-300 mt-4">
                    <?php if ($blockedReason === 'not_entitled'): ?>
                        <p>Your access code does not include this content.</p>
                    <?php elseif ($blockedReason === 'out_of_window'): ?>
                        <p>This content is not currently available. Please check the schedule.</p>
                    <?php else: ?>
                        <p>Playback is temporarily unavailable. Please try again later.</p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>

            <!-- YouTube Player Initialization Script -->
            <script>
                // Pass the dynamic video ID to the player script
                window.videoId = '<?php echo htmlspecialchars($videoUrl); ?>';
                window.isAdmin = <?php echo isset($_SESSION['is_admin']) ? 'true' : 'false'; ?>;
                function onYouTubeIframeAPIReady() {
                    // initPlayer is defined in js/player.js
                    initPlayer(window.videoId);
                }
            </script>

            <?php if (isset($_SESSION['is_admin'])): ?>
                <!-- ADMIN PANEL -->
                <div class="mt-8 p-6 bg-gray-800 rounded-xl shadow-lg w-full border border-gray-700">
                    <h2 class="text-xl font-bold text-yellow-400 mb-4 flex items-center">
                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"></path></svg> Admin Panel
                    </h2>
                    
                    <!-- Set Premium Video Content -->
                    <div class="mb-6 border-b border-gray-700 pb-4">
                        <h3 class="text-lg font-semibold text-white mb-2">Set Premium Video Content</h3>
                        <p class="text-sm text-gray-400 mb-3">Current Video ID: <span class="font-mono font-bold text-yellow-300"><?php echo htmlspecialchars($videoUrl); ?></span></p>
                        <!-- Form now correctly targets admin.php -->
                        <form method="POST" action="admin.php">
                            <div class="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-2">
                                <input type="text" name="admin_video_url" value="<?php echo htmlspecialchars($videoUrl); ?>" placeholder="YouTube Video ID or full URL" class="flex-grow p-2 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500 outline-none transition">
                                <button type="submit" name="save_video" class="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg transition duration-200">Update Video</button>
                            </div>
                        </form>
                    </div>

                    <!-- Manage Access Codes (Add Single Code) -->
                    <h3 class="text-lg font-semibold text-white mb-3">Manage Access Codes (<?php echo count($codes); ?>)</h3>
                    <!-- Form now correctly targets admin.php -->
                    <form method="POST" action="admin.php" class="flex space-x-2 mb-4 border-b border-gray-700 pb-4">
                        <input type="text" name="new_code" maxlength="10" placeholder="New Code (max 10 chars)" class="flex-grow p-2 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500 outline-none transition">
                        <button type="submit" name="add_code" class="px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg flex items-center transition duration-200">
                            <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path></svg> Add Code
                        </button>
                    </form>

                    <!-- Bulk Upload Codes (CSV) -->
                    <div class="mb-6 pt-2">
                        <h4 class="text-md font-semibold text-gray-300 mb-2 flex items-center">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-3 7h3m-3 4h3m-6-4h.01M9 16h.01"></path></svg> Bulk Upload Codes (CSV)
                        </h4>
                        <p class="text-xs text-gray-400 mb-3">Upload a file with one access code per line. Max 10 characters per code.</p>
                        <!-- Form now correctly targets admin.php -->
                        <form method="POST" action="admin.php" enctype="multipart/form-data">
                            <input type="file" name="csv_file" accept=".csv,.txt" class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-purple-50 file:text-purple-700 hover:file:bg-purple-100">
                            <button type="submit" name="upload_csv" class="mt-2 px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded-lg">Upload</button>
                        </form>
                    </div>
                    
                    <!-- Access Code List -->
                    <div class="max-h-60 overflow-y-auto pr-2 border-t border-gray-700 pt-4">
                        <?php foreach ($codes as $codeRecord): ?>
                            <div class="flex justify-between items-center p-3 my-2 bg-gray-700 rounded-lg">
                                <span class="font-mono font-bold <?php echo $codeRecord['used_by'] ? 'text-green-300' : 'text-gray-300'; ?>">
                                    <?php echo htmlspecialchars($codeRecord['code']); ?>
                                </span>
                                <div class="text-xs text-right">
                                    <?php if ($codeRecord['used_by']): ?>
                                        <p class="text-green-400 flex items-center justify-end">
                                            <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg> Used
                                        </p>
                                        <p class="text-gray-400 break-all">FP: <?php echo htmlspecialchars(substr($codeRecord['used_by'], 0, 10)); ?>...</p>
                                    <?php else: ?>
                                        <p class="text-red-400">Unused</p>
                                    <?php endif; ?>
                                </div>
                                <!-- Delete Form now correctly targets admin.php -->
                                <form method="POST" action="admin.php">
                                    <input type="hidden" name="delete_code" value="<?php echo htmlspecialchars($codeRecord['code']); ?>">
                                    <button type="submit" class="p-2 bg-red-600 hover:bg-red-700 text-white rounded-full transition duration-200" title="Delete Code">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg>
                                    </button>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <!-- LOGGED-OUT VIEW (LOGIN FORM) -->
            <div class="flex flex-col items-center mt-8 w-full max-w-sm">
                <!-- Centered Logo Above Login Box -->
                <div class="mb-6 flex justify-center">
                    <img src="img/chazon.png" alt="Site Logo" class="h-20 w-auto select-none" loading="eager">
                </div>
                
                <!-- Login Box -->
                <div class="p-8 bg-gray-800 rounded-xl shadow-lg w-full border border-purple-500/50">
                    <div class="flex flex-col items-center mb-6">
                        <svg class="w-8 h-8 text-purple-400 mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path></svg>
                        <h2 class="text-2xl font-bold text-white">Exclusive Content</h2>
                        <p class="text-gray-400 text-sm mt-1">Enter your one-time access code.</p>
                    </div>
                <!-- Form targets the same index.php file to handle login POST -->
                <form method="POST">
                    <input type="text" name="access_code" placeholder="Enter Access Code" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 mb-4 text-center font-mono focus:ring-purple-500 focus:border-purple-500 outline-none transition" maxlength="10">
                    <button type="submit" class="w-full py-3 bg-purple-600 hover:bg-purple-700 text-white font-semibold rounded-lg shadow-md transition duration-200 transform hover:scale-[1.01]">
                        <svg class="w-4 h-4 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"></path></svg> Access Video
                    </button>
                </form>
                <p class="text-center text-gray-500 text-xs mt-4">
                    Device ID: <span class="font-mono"><?php echo htmlspecialchars(substr($userFingerprint, 0, 15)); ?>...</span>
                </p>
                </div>
            </div>
        <?php endif; ?> 
    </main>

    <!-- Details Modal -->
    <div id="details-modal" class="fixed inset-0 bg-black/60 hidden items-center justify-center z-50" aria-modal="true" role="dialog">
        <div class="bg-gray-900 border border-gray-700 rounded-lg max-w-lg w-[90%] overflow-hidden shadow-2xl">
            <div class="flex justify-between items-center px-4 py-3 border-b border-gray-700">
                <h4 id="dm-title" class="text-white text-lg font-semibold"></h4>
                <button id="dm-close" class="text-gray-300 hover:text-white" aria-label="Close details">✕</button>
            </div>
            <div class="p-4">
                <div id="dm-artwrap" class="mb-3 hidden">
                    <img id="dm-art" src="" alt="Artwork" class="w-full h-40 object-cover rounded" />
                </div>
                <div id="dm-badge" class="inline-block text-xs px-2 py-1 rounded mb-3"></div>
                <p id="dm-synopsis" class="text-gray-200 text-sm leading-relaxed"></p>
                <p id="dm-tags" class="text-xs text-gray-400 mt-3"></p>
                <div class="mt-4">
                    <a id="dm-play" href="#" class="inline-flex items-center px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white rounded">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>
                        Play
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script>
        (function(){
            function $(s){ return document.querySelector(s); }
            function fmt(dt){ try { return new Date(dt.replace(' ', 'T')).toLocaleString(); } catch(_) { return dt; } }
            const modal = $('#details-modal');
            const titleEl = $('#dm-title');
            const synEl = $('#dm-synopsis');
            const badge = $('#dm-badge');
            const artwrap = $('#dm-artwrap');
            const art = $('#dm-art');
            const tagsEl = $('#dm-tags');
            const playBtn = $('#dm-play');
            const closeBtn = $('#dm-close');
            let lastFocus = null;
            function openModal(data){
                lastFocus = document.activeElement;
                titleEl.textContent = data.title || '';
                synEl.textContent = data.synopsis || 'No description available.';
                // Badge
                badge.className = 'inline-block text-xs px-2 py-1 rounded mb-3';
                if (data.live === '1') {
                    badge.textContent = 'Live now';
                    badge.classList.add('bg-green-600','text-white');
                } else if (data.nextstart) {
                    badge.textContent = 'Starts at ' + fmt(data.nextstart);
                    badge.classList.add('bg-yellow-600','text-white');
                } else {
                    badge.textContent = 'Not currently available';
                    badge.classList.add('bg-gray-600','text-white');
                }
                if (data.art) {
                    artwrap.classList.remove('hidden');
                    art.src = data.art;
                } else {
                    artwrap.classList.add('hidden');
                    art.src = '';
                }
                // Tags
                if (data.tags) {
                    tagsEl.textContent = 'Tags: ' + data.tags;
                    tagsEl.classList.remove('hidden');
                } else {
                    tagsEl.textContent = '';
                    tagsEl.classList.add('hidden');
                }
                // Play link
                if (data.yt) { playBtn.href = '?v=' + encodeURIComponent(data.yt); }
                modal.classList.remove('hidden');
                modal.classList.add('flex');
                // Focus trap start on close button
                closeBtn.focus();
            }
            function closeModal(){ modal.classList.add('hidden'); modal.classList.remove('flex'); if (lastFocus) { lastFocus.focus(); } }
            document.addEventListener('click', function(e){
                const t = e.target;
                if (t && t.classList && t.classList.contains('details-btn')){
                    e.preventDefault();
                    openModal({
                        title: t.getAttribute('data-title') || '',
                        synopsis: t.getAttribute('data-synopsis') || '',
                        art: t.getAttribute('data-art') || '',
                        live: t.getAttribute('data-live') || '0',
                        nextstart: t.getAttribute('data-nextstart') || '',
                        yt: t.getAttribute('data-yt') || '',
                        tags: t.getAttribute('data-tags') || ''
                    });
                }
            });
            closeBtn.addEventListener('click', closeModal);
            $('#details-modal').addEventListener('click', function(e){ if (e.target === this) closeModal(); });
            // ESC to close & basic focus trap
            document.addEventListener('keydown', function(e){
                if (modal.classList.contains('hidden')) return;
                if (e.key === 'Escape') { e.preventDefault(); closeModal(); }
                if (e.key === 'Tab') {
                    const focusables = [closeBtn, playBtn];
                    const idx = focusables.indexOf(document.activeElement);
                    if (e.shiftKey) {
                        if (idx <= 0) { e.preventDefault(); focusables[focusables.length - 1].focus(); }
                    } else {
                        if (idx === focusables.length - 1) { e.preventDefault(); focusables[0].focus(); }
                    }
                }
            });
        })();
    </script>
    <script>
        (function() {
            // Disable context menu (right-click)
            window.addEventListener('contextmenu', function(e) { e.preventDefault(); }, { capture: true });

            // Disable common devtools and player/scroll shortcuts
            window.addEventListener('keydown', function(e) {
                const key = (e.key || '').toUpperCase();
                const meta = e.ctrlKey || e.metaKey;
                const active = document.activeElement;
                const tag = (active && active.tagName) ? active.tagName.toLowerCase() : '';
                const isEditable = (active && (active.isContentEditable || tag === 'input' || tag === 'textarea' || tag === 'select'));
                if (isEditable) { return; }

                // Devtools / view-source / save / print
                if (key === 'F12') { e.preventDefault(); return; }
                if (meta && e.shiftKey && (key === 'I' || key === 'J' || key === 'C')) { e.preventDefault(); return; }
                if (meta && (key === 'U' || key === 'S' || key === 'P')) { e.preventDefault(); return; }

                // Block common playback/scroll keys (YouTube shortcuts & arrows/space)
                const blockKeys = [' ', 'ARROWLEFT', 'ARROWRIGHT', 'ARROWUP', 'ARROWDOWN', 'K', 'J', 'L', 'M'];
                if (blockKeys.includes(key)) { e.preventDefault(); return; }
            }, { capture: true });
        })();
    </script>
    
    <!-- Footer with legal links -->
    <footer class="mt-12 py-6 border-t border-gray-800 text-center text-sm text-gray-500">
        <div class="max-w-7xl mx-auto px-4">
            <p class="mb-2">&copy; <?= date('Y') ?> CET Networks. All rights reserved.</p>
            <div class="flex justify-center gap-6 flex-wrap">
                <a href="terms.php" class="hover:text-purple-400 transition">Terms of Service</a>
                <a href="privacy.php" class="hover:text-purple-400 transition">Privacy Policy</a>
                <a href="gdpr.php" class="hover:text-purple-400 transition">GDPR Data Request</a>
                <?php if (isset($_SESSION['is_admin'])): ?>
                <a href="forgot_admin.php" class="hover:text-purple-400 transition">Reset Admin Code</a>
                <?php endif; ?>
            </div>
        </div>
    </footer>
</body>
</html>
