<?php
// GDPR Data Request Portal
ini_set('session.gc_maxlifetime', 7200);
session_set_cookie_params([
    'lifetime' => 7200,
    'httponly' => true,
    'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443),
    'samesite' => 'Lax'
]);
session_start();
require_once __DIR__ . '/../config/db.php';

$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $code = strtoupper(trim($_POST['code'] ?? ''));
    
    if ($code === '' || strlen($code) > 10) {
        $message = 'Invalid access code';
        $messageType = 'error';
    } else {
        try {
            // Verify code exists
            $stmt = $pdo->prepare("SELECT code FROM access_codes WHERE code = ? LIMIT 1");
            $stmt->execute([$code]);
            if (!$stmt->fetchColumn()) {
                $message = 'Access code not found';
                $messageType = 'error';
            } else {
                if ($action === 'export') {
                    // Export user data
                    $data = [];
                    
                    // Access code info
                    $stmt = $pdo->prepare("SELECT code, used_by, created_at FROM access_codes WHERE code = ?");
                    $stmt->execute([$code]);
                    $data['access_code'] = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    // Sessions
                    $stmt = $pdo->prepare("SELECT session_id, fingerprint, last_activity FROM sessions WHERE code = ?");
                    $stmt->execute([$code]);
                    $data['sessions'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Profiles
                    $stmt = $pdo->prepare("SELECT id, name, created_at FROM profiles WHERE code = ?");
                    $stmt->execute([$code]);
                    $data['profiles'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Video views
                    $stmt = $pdo->prepare("SELECT v.youtube_id, vv.started_at, vv.watch_duration, vv.completed 
                                           FROM video_views vv 
                                           JOIN videos v ON v.id = vv.video_id 
                                           WHERE vv.code = ?");
                    $stmt->execute([$code]);
                    $data['viewing_history'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Send as JSON download
                    header('Content-Type: application/json');
                    header('Content-Disposition: attachment; filename=gdpr_data_' . $code . '_' . date('Ymd') . '.json');
                    echo json_encode($data, JSON_PRETTY_PRINT);
                    exit;
                    
                } elseif ($action === 'delete') {
                    // Delete all user data
                    $pdo->beginTransaction();
                    try {
                        $pdo->prepare("DELETE FROM sessions WHERE code = ?")->execute([$code]);
                        $pdo->prepare("DELETE FROM profiles WHERE code = ?")->execute([$code]);
                        $pdo->prepare("DELETE FROM video_views WHERE code = ?")->execute([$code]);
                        $pdo->prepare("DELETE FROM code_entitlements WHERE code = ?")->execute([$code]);
                        $pdo->prepare("DELETE FROM access_codes WHERE code = ?")->execute([$code]);
                        $pdo->commit();
                        
                        $message = 'All data associated with code ' . htmlspecialchars($code) . ' has been permanently deleted.';
                        $messageType = 'success';
                    } catch (PDOException $e) {
                        $pdo->rollBack();
                        $message = 'Error deleting data: ' . $e->getMessage();
                        $messageType = 'error';
                    }
                }
            }
        } catch (PDOException $e) {
            $message = 'Database error: ' . $e->getMessage();
            $messageType = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GDPR Data Request - PPV System</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-gray-200">
    <div class="max-w-2xl mx-auto px-4 py-12">
        <h1 class="text-4xl font-bold text-white mb-4">GDPR Data Request</h1>
        <p class="text-gray-400 mb-8">Exercise your data rights under GDPR</p>
        
        <?php if ($message): ?>
        <div class="mb-6 p-4 rounded-lg <?= $messageType === 'error' ? 'bg-red-900/50 border border-red-600 text-red-200' : 'bg-green-900/50 border border-green-600 text-green-200' ?>">
            <?= $message ?>
        </div>
        <?php endif; ?>
        
        <div class="bg-gray-800 p-6 rounded-lg border border-gray-700 mb-6">
            <h2 class="text-2xl font-semibold text-white mb-4">Export Your Data</h2>
            <p class="text-gray-300 mb-4">Download all data associated with your access code in JSON format.</p>
            <form method="POST" class="space-y-4">
                <input type="hidden" name="action" value="export">
                <div>
                    <label class="block text-sm font-medium mb-2">Access Code</label>
                    <input type="text" name="code" maxlength="10" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500" placeholder="Enter your access code" required>
                </div>
                <button type="submit" class="w-full py-3 bg-purple-600 hover:bg-purple-700 text-white font-semibold rounded-lg transition">
                    Download My Data
                </button>
            </form>
        </div>
        
        <div class="bg-gray-800 p-6 rounded-lg border border-red-600/50">
            <h2 class="text-2xl font-semibold text-white mb-4">Delete Your Data</h2>
            <p class="text-gray-300 mb-4"><strong class="text-red-400">Warning:</strong> This will permanently delete all data associated with your access code, including:</p>
            <ul class="list-disc ml-6 text-gray-300 mb-4 space-y-1">
                <li>Access code and authentication data</li>
                <li>All profiles</li>
                <li>Viewing history and analytics</li>
                <li>Session data</li>
                <li>Video entitlements</li>
            </ul>
            <p class="text-red-400 font-semibold mb-4">This action cannot be undone!</p>
            <form method="POST" onsubmit="return confirm('Are you absolutely sure? This will permanently delete ALL your data and cannot be undone.');">
                <input type="hidden" name="action" value="delete">
                <div class="mb-4">
                    <label class="block text-sm font-medium mb-2">Access Code</label>
                    <input type="text" name="code" maxlength="10" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-red-500 focus:border-red-500" placeholder="Enter your access code" required>
                </div>
                <button type="submit" class="w-full py-3 bg-red-600 hover:bg-red-700 text-white font-semibold rounded-lg transition">
                    Permanently Delete My Data
                </button>
            </form>
        </div>
        
        <div class="mt-8 pt-6 border-t border-gray-700 flex justify-between text-sm">
            <a href="privacy.php" class="text-purple-400 hover:text-purple-300">← Privacy Policy</a>
            <a href="index.php" class="text-purple-400 hover:text-purple-300">Back to Home →</a>
        </div>
    </div>
</body>
</html>
