<?php
// Admin Password Reset Request
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/email.php';

$message = '';
$messageType = '';

// Ensure reset tokens table exists
try {
    $pdo->exec("CREATE TABLE IF NOT EXISTS admin_reset_tokens (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        email VARCHAR(255) NOT NULL,
        token VARCHAR(64) NOT NULL UNIQUE,
        expires_at TIMESTAMP NOT NULL,
        used TINYINT(1) DEFAULT 0,
        created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        INDEX idx_token (token),
        INDEX idx_expires (expires_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
} catch (PDOException $e) {}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $message = 'Invalid email address';
        $messageType = 'error';
    } else {
        // Check if email is registered as admin email
        try {
            $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'admin_email' LIMIT 1");
            $stmt->execute();
            $adminEmail = $stmt->fetchColumn();
            
            if ($adminEmail && $adminEmail === $email) {
                // Generate reset token
                $token = bin2hex(random_bytes(32));
                $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour'));
                
                $stmt = $pdo->prepare("INSERT INTO admin_reset_tokens (email, token, expires_at) VALUES (?, ?, ?)");
                $stmt->execute([$email, $token, $expiresAt]);
                
                // Send email
                $emailService = new EmailService($pdo);
                $emailService->sendAdminResetLink($email, $token);
                
                $message = 'Reset link sent to your email. Check your inbox.';
                $messageType = 'success';
            } else {
                // Don't reveal if email exists or not (security)
                $message = 'If this email is registered, you will receive a reset link shortly.';
                $messageType = 'success';
            }
        } catch (PDOException $e) {
            $message = 'Error processing request. Please try again.';
            $messageType = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Admin Code - PPV System</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-900 text-gray-200 min-h-screen flex items-center justify-center">
    <div class="max-w-md w-full px-4">
        <div class="bg-gray-800 p-8 rounded-xl border border-purple-500/50 shadow-2xl">
            <h1 class="text-3xl font-bold text-white mb-2">Forgot Admin Code?</h1>
            <p class="text-gray-400 mb-6">Enter your admin email to receive a reset link</p>
            
            <?php if ($message): ?>
            <div class="mb-6 p-4 rounded-lg <?= $messageType === 'error' ? 'bg-red-900/50 border border-red-600 text-red-200' : 'bg-green-900/50 border border-green-600 text-green-200' ?>">
                <?= htmlspecialchars($message) ?>
            </div>
            <?php endif; ?>
            
            <form method="POST" class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Admin Email</label>
                    <input type="email" name="email" class="w-full p-3 rounded-lg bg-gray-900 text-white border border-gray-700 focus:ring-purple-500 focus:border-purple-500" placeholder="admin@example.com" required>
                </div>
                <button type="submit" class="w-full py-3 bg-purple-600 hover:bg-purple-700 text-white font-semibold rounded-lg transition">
                    Send Reset Link
                </button>
            </form>
            
            <div class="mt-6 pt-6 border-t border-gray-700 text-center">
                <a href="index.php" class="text-purple-400 hover:text-purple-300 text-sm">← Back to Login</a>
            </div>
            
            <div class="mt-4 p-4 bg-gray-900 rounded-lg">
                <p class="text-xs text-gray-400">
                    <strong>Note:</strong> You must have set an admin email in the system settings for this to work. 
                    If you haven't set one, please contact your system administrator.
                </p>
            </div>
        </div>
    </div>
</body>
</html>
