<?php
// Utility functions for session, messages, and fingerprinting

/**
 * Generates a simple, unique user fingerprint based on browser/client info.
 * NOTE: This is for demonstration. A robust fingerprint needs client-side JS.
 * @return string A SHA256 hash of user agent and IP address.
 */
function generateFingerprint() {
    // Note: REMOTE_ADDR can change frequently. User Agent is slightly more stable.
    return hash('sha256', $_SERVER['HTTP_USER_AGENT'] . ($_SERVER['REMOTE_ADDR'] ?? 'unknown_ip'));
}

/**
 * Sets a session message (error or success) for display after redirect.
 * @param string $text The message content.
 * @param bool $isError True for error, false for success/normal message.
 */
function showMessage($text, $isError = false) {
    if ($isError) {
        $_SESSION['error'] = $text;
    } else {
        $_SESSION['message'] = $text;
    }
}

/**
 * Retrieves and clears a session message.
 * @param string $type 'error' or 'message'.
 * @return string|null The message content or null if not set.
 */
function getMessage($type) {
    if (isset($_SESSION[$type])) {
        $message = $_SESSION[$type];
        unset($_SESSION[$type]);
        return $message;
    }
    return null;
}

/**
 * Returns active session TTL in minutes from env or default 30.
 */
function getActiveSessionTTL() {
    $ttl = getenv('ACTIVE_SESSION_TTL_MINUTES');
    return ($ttl && is_numeric($ttl) && (int)$ttl > 0) ? (int)$ttl : 30;
}

/**
 * Logs an audit event into audit_logs table.
 * @param PDO $pdo
 * @param string $eventType e.g., login_success, login_fail, logout, taken_over
 * @param string $details optional textual details
 */
function logAudit($pdo, $eventType, $details = '') {
    try {
        $code = isset($_SESSION['current_code']) ? $_SESSION['current_code'] : null;
        $sid = session_id();
        $fp = generateFingerprint();
        $stmt = $pdo->prepare("INSERT INTO audit_logs (event_type, code, session_id, fingerprint, details) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$eventType, $code, $sid, $fp, $details]);
    } catch (PDOException $e) {
        error_log('Audit log error: ' . $e->getMessage());
    }
}

/**
 * Validates if the current session is still active and valid in the database.
 * This checks if another device has taken over the code.
 * @param PDO $pdo
 * @param string $fingerprint The user's device fingerprint.
 * @return bool True if active and matching, false otherwise.
 */
function validateActiveSession($pdo, $fingerprint) {
    if (!isset($_SESSION['current_code']) || !isset($_SESSION['is_logged_in'])) {
        return false;
    }

    try {
        // Sliding window for activity timeout (default 30 minutes)
        $ttl = getenv('ACTIVE_SESSION_TTL_MINUTES');
        $ttl = ($ttl && is_numeric($ttl) && (int)$ttl > 0) ? (int)$ttl : 30;
        // NOTE: Avoid binding inside INTERVAL; embed validated integer
        $sql = "UPDATE active_sessions SET last_activity = NOW() WHERE code = ? AND session_id = ? AND last_activity > (NOW() - INTERVAL $ttl MINUTE)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$_SESSION['current_code'], session_id()]);
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Session validation database error: " . $e->getMessage());
        // If DB fails, assume inactive session for safety
        return false;
    }
}

/**
 * Invalidates any old session entries for a given code and sets the new one.
 * This ensures only one device can use a code simultaneously.
 * @param PDO $pdo
 * @param string $code The access code used.
 * @param string $fingerprint The new device fingerprint.
 * @param string $sessionId The current PHP session ID.
 */
function invalidateOldSession($pdo, $code, $fingerprint, $sessionId) {
    try {
        $pdo->beginTransaction();
        // 1. Delete any existing sessions for this code (only one allowed at a time)
        $stmt = $pdo->prepare("DELETE FROM active_sessions WHERE code = ?");
        $stmt->execute([$code]);
        // 2. Insert the new active session
        $insertStmt = $pdo->prepare("
            INSERT INTO active_sessions (code, fingerprint, session_id, last_activity)
            VALUES (?, ?, ?, NOW())
        ");
        $insertStmt->execute([$code, $fingerprint, $sessionId]);
        $pdo->commit();
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) { $pdo->rollBack(); }
        error_log("Error invalidating old session: " . $e->getMessage());
    }
}
