<?php
// Simple SMS Service Wrapper (configure your provider)
class SMSService {
    private $provider;
    private $apiKey;
    private $senderId;

    public function __construct() {
        $this->provider = getenv('SMS_PROVIDER') ?: 'twilio'; // 'twilio' or 'termii' or 'generic'
        $this->apiKey = getenv('SMS_API_KEY') ?: '';
        $this->senderId = getenv('SMS_SENDER_ID') ?: 'PPV';
    }

    public function send($phone, $message) {
        $phone = trim($phone);
        if ($phone === '' || $message === '') return false;

        try {
            if ($this->provider === 'termii') {
                return $this->sendViaTermii($phone, $message);
            } elseif ($this->provider === 'twilio') {
                return $this->sendViaTwilio($phone, $message);
            }
            return $this->sendViaGeneric($phone, $message);
        } catch (Throwable $e) {
            error_log('SMS send failed: ' . $e->getMessage());
            return false;
        }
    }

    private function sendViaTermii($phone, $message) {
        $endpoint = 'https://api.ng.termii.com/api/sms/send';
        $payload = [
            'to' => $phone,
            'from' => $this->senderId,
            'sms' => $message,
            'type' => 'plain',
            'api_key' => $this->apiKey,
            'channel' => 'generic'
        ];
        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $resp = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return $http >= 200 && $http < 300;
    }

    private function sendViaTwilio($phone, $message) {
        // Minimal Twilio Send SMS via API
        $sid = getenv('TWILIO_SID') ?: '';
        $token = getenv('TWILIO_TOKEN') ?: '';
        $from = getenv('TWILIO_FROM') ?: '';
        if (!$sid || !$token || !$from) return false;
        $url = "https://api.twilio.com/2010-04-01/Accounts/{$sid}/Messages.json";
        $fields = http_build_query(['From' => $from, 'To' => $phone, 'Body' => $message]);
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_USERPWD, $sid . ':' . $token);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $resp = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return $http >= 200 && $http < 300;
    }

    private function sendViaGeneric($phone, $message) {
        // Placeholder: implement your own SMS gateway here
        error_log("[SMS GENERIC] To: {$phone} Msg: {$message}");
        return true;
    }
}
