<?php
// Email Notification System
class EmailService {
    private $pdo;
    private $from;
    private $fromName;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->from = getenv('MAIL_FROM') ?: 'noreply@yoursite.com';
        $this->fromName = getenv('MAIL_FROM_NAME') ?: 'PPV System';
        
        // Ensure email_notifications table exists
        try {
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS email_notifications (
                id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
                recipient VARCHAR(255) NOT NULL,
                subject VARCHAR(255) NOT NULL,
                body TEXT NOT NULL,
                sent_at TIMESTAMP NULL,
                status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
                error_message TEXT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                INDEX idx_status (status),
                INDEX idx_created (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
        } catch (PDOException $e) {}
    }
    
    public function send($to, $subject, $body, $isHtml = true) {
        // Queue email
        try {
            $stmt = $this->pdo->prepare("INSERT INTO email_notifications (recipient, subject, body) VALUES (?, ?, ?)");
            $stmt->execute([$to, $subject, $body]);
            $emailId = $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Failed to queue email: " . $e->getMessage());
            return false;
        }
        
        // Send email
        $headers = [
            'From' => "{$this->fromName} <{$this->from}>",
            'Reply-To' => $this->from,
            'X-Mailer' => 'PHP/' . phpversion(),
            'MIME-Version' => '1.0'
        ];
        
        if ($isHtml) {
            $headers['Content-Type'] = 'text/html; charset=UTF-8';
        } else {
            $headers['Content-Type'] = 'text/plain; charset=UTF-8';
        }
        
        $headerString = '';
        foreach ($headers as $key => $value) {
            $headerString .= "$key: $value\r\n";
        }
        
        $success = mail($to, $subject, $body, $headerString);
        
        // Update status
        try {
            if ($success) {
                $stmt = $this->pdo->prepare("UPDATE email_notifications SET status = 'sent', sent_at = NOW() WHERE id = ?");
                $stmt->execute([$emailId]);
            } else {
                $stmt = $this->pdo->prepare("UPDATE email_notifications SET status = 'failed', error_message = 'mail() returned false' WHERE id = ?");
                $stmt->execute([$emailId]);
            }
        } catch (PDOException $e) {}
        
        return $success;
    }
    
    public function sendNewDeviceAlert($email, $code, $deviceInfo) {
        $subject = "New Device Login Alert - PPV System";
        $body = "
        <html>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <h2 style='color: #6B46C1;'>New Device Login Detected</h2>
            <p>Your access code <strong>{$code}</strong> was just used on a new device.</p>
            <p><strong>Device Information:</strong><br>{$deviceInfo}</p>
            <p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>
            <p>If this wasn't you, please contact support immediately.</p>
            <hr style='border: 1px solid #eee; margin: 20px 0;'>
            <p style='font-size: 12px; color: #666;'>This is an automated message. Please do not reply.</p>
        </body>
        </html>
        ";
        
        return $this->send($email, $subject, $body, true);
    }
    
    public function sendAdminResetLink($email, $token) {
        $resetLink = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . 
                     "://{$_SERVER['HTTP_HOST']}/public/reset_admin.php?token={$token}";
        
        $subject = "Admin Code Reset Request - PPV System";
        $body = "
        <html>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <h2 style='color: #6B46C1;'>Admin Code Reset Request</h2>
            <p>You requested to reset your admin code.</p>
            <p>Click the link below to reset your code (valid for 1 hour):</p>
            <p><a href='{$resetLink}' style='display: inline-block; padding: 10px 20px; background-color: #6B46C1; color: white; text-decoration: none; border-radius: 5px;'>Reset Admin Code</a></p>
            <p>Or copy this link: <br><code style='background: #f4f4f4; padding: 5px;'>{$resetLink}</code></p>
            <p>If you didn't request this, please ignore this email.</p>
            <hr style='border: 1px solid #eee; margin: 20px 0;'>
            <p style='font-size: 12px; color: #666;'>This is an automated message. Please do not reply.</p>
        </body>
        </html>
        ";
        
        return $this->send($email, $subject, $body, true);
    }
}
