
<?php
// admin.php

// -----------------------------------------------------
// 1. INITIAL SETUP & ERROR HANDLING
// -----------------------------------------------------
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
session_start();

// Basic security headers (adjust for your deployment)
header("Referrer-Policy: no-referrer");
header("X-Frame-Options: SAMEORIGIN");
header("X-Content-Type-Options: nosniff");
header("Permissions-Policy: fullscreen=(self)");
header("Content-Security-Policy: default-src 'self'; img-src 'self' data: https:; script-src 'self' https://cdn.tailwindcss.com; style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; connect-src 'self'; font-src 'self' https://fonts.gstatic.com;");

// Robust includes using absolute paths
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/utils.php';

// Restrict to admin only
if (!isset($_SESSION['is_admin']) || $_SESSION['is_admin'] !== true) {
    showMessage('Unauthorized access.', true);
    header('Location: index.php');
    exit;
}

// Handle delete entitlement
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_entitlement'])) {
    $codeDel = strtoupper(trim($_POST['del_ent_code'] ?? ''));
    $videoDel = (int)($_POST['del_ent_video_id'] ?? 0);
    if ($codeDel && $videoDel > 0) {
        try {
            $stmt = $pdo->prepare("DELETE FROM code_entitlements WHERE code = ? AND video_id = ?");
            $stmt->execute([$codeDel, $videoDel]);
            showMessage('Entitlement removed.', false);
        } catch (PDOException $e) {
            showMessage('Error removing entitlement: ' . $e->getMessage(), true);
        }
    }
    $redirect_needed = true;
}

// Handle delete schedule
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_schedule'])) {
    $schId = (int)($_POST['del_sch_id'] ?? 0);
    if ($schId > 0) {
        try {
            $stmt = $pdo->prepare("DELETE FROM schedules WHERE id = ?");
            $stmt->execute([$schId]);
            showMessage('Schedule removed.', false);
        } catch (PDOException $e) {
            showMessage('Error removing schedule: ' . $e->getMessage(), true);
        }
    }
    $redirect_needed = true;
}

// Handle create video entry
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_video'])) {
    $youtubeId = trim($_POST['youtube_id'] ?? '');
    $title = trim($_POST['video_title'] ?? '');
    $synopsis = trim($_POST['video_synopsis'] ?? '');
    $art = trim($_POST['artwork_url'] ?? '');
    $tags = trim($_POST['tags'] ?? '');
    if (!preg_match('/^[a-zA-Z0-9_-]{11}$/', $youtubeId) || $title === '') {
        showMessage('Provide valid YouTube ID (11 chars) and a title.', true);
        $redirect_needed = true;
    } else {
        try {
            $stmt = $pdo->prepare("INSERT INTO videos (youtube_id, title, synopsis, artwork_url, tags) VALUES (?,?,?,?,?)");
            $stmt->execute([$youtubeId, $title, $synopsis ?: null, $art ?: null, $tags ?: null]);
            showMessage('Video added.', false);
        } catch (PDOException $e) {
            showMessage('Error adding video: ' . $e->getMessage(), true);
        }
        $redirect_needed = true;
    }
}

// Handle add entitlement code->video
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_entitlement'])) {
    $codeEnt = strtoupper(trim($_POST['ent_code'] ?? ''));
    $videoIdEnt = (int)($_POST['ent_video_id'] ?? 0);
    if ($codeEnt === '' || $videoIdEnt <= 0) {
        showMessage('Select a video and enter a code.', true);
    } else {
        try {
            $stmt = $pdo->prepare("INSERT IGNORE INTO code_entitlements (code, video_id) VALUES (?, ?)");
            $stmt->execute([$codeEnt, $videoIdEnt]);
            showMessage('Entitlement added.', false);
        } catch (PDOException $e) {
            showMessage('Error adding entitlement: ' . $e->getMessage(), true);
        }
    }
    $redirect_needed = true;
}

// Handle add schedule
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_schedule'])) {
    $vid = (int)($_POST['sch_video_id'] ?? 0);
    $start = trim($_POST['starts_at'] ?? '');
    $end = trim($_POST['ends_at'] ?? '');
    if ($vid <= 0 || $start === '' || $end === '') {
        showMessage('Provide video and time window.', true);
    } else {
        try {
            $stmt = $pdo->prepare("INSERT INTO schedules (video_id, starts_at, ends_at) VALUES (?,?,?)");
            $stmt->execute([$vid, $start, $end]);
            showMessage('Schedule added.', false);
        } catch (PDOException $e) {
            showMessage('Error adding schedule: ' . $e->getMessage(), true);
        }
    }
    $redirect_needed = true;
}

// Handle hero settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_hero'])) {
    $heroTitle = trim($_POST['hero_title'] ?? '');
    $heroSynopsis = trim($_POST['hero_synopsis'] ?? '');
    $heroImage = trim($_POST['hero_image'] ?? '');
    try {
        $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES ('hero_title', ?)");
        $stmt->execute([$heroTitle]);
        $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES ('hero_synopsis', ?)");
        $stmt->execute([$heroSynopsis]);
        $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES ('hero_image', ?)");
        $stmt->execute([$heroImage]);
        showMessage('Hero settings saved.', false);
    } catch (PDOException $e) {
        showMessage('Error saving hero settings: ' . $e->getMessage(), true);
    }
    $redirect_needed = true;
}

// -----------------------------------------------------
// 2. HANDLE POST REQUESTS (The logic you provided)
// -----------------------------------------------------

$redirect_needed = false;

// Handle video update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_video'])) {
    $videoUrl = trim($_POST['admin_video_url']);
    // Extract video ID from full URL or use as-is
    if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/', $videoUrl, $match)) {
        $videoUrl = $match[1];
    }
    if (strlen($videoUrl) === 11 && preg_match('/^[a-zA-Z0-9_-]{11}$/', $videoUrl)) {
        try {
            $stmt = $pdo->prepare("REPLACE INTO settings (setting_key, setting_value) VALUES ('active_video_id', ?)");
            $stmt->execute([$videoUrl]);
            showMessage('Video updated successfully.', false);
        } catch (PDOException $e) {
            showMessage('Error updating video: ' . $e->getMessage(), true);
        }
    } else {
        showMessage('Invalid YouTube video ID.', true);
    }
    $redirect_needed = true;
}

// Handle new code
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_code'])) {
    $newCode = strtoupper(trim($_POST['new_code']));
    if (strlen($newCode) <= 10 && preg_match('/^[A-Z0-9]+$/', $newCode)) {
        try {
            // Check for duplicate code before insert
            $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM access_codes WHERE code = ?");
            $checkStmt->execute([$newCode]);
            if ($checkStmt->fetchColumn() > 0) {
                 showMessage('Code already exists.', true);
            } else {
                $stmt = $pdo->prepare("INSERT INTO access_codes (code) VALUES (?)");
                $stmt->execute([$newCode]);
                showMessage('Code added successfully.', false);
            }
        } catch (PDOException $e) {
            showMessage('Error adding code: ' . $e->getMessage(), true);
        }
    } else {
        showMessage('Invalid code format. Use up to 10 alphanumeric characters.', true);
    }
    $redirect_needed = true;
}

// Handle CSV upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_csv']) && isset($_FILES['csv_file'])) {
    $file = $_FILES['csv_file'];
    if ($file['type'] === 'text/csv' || $file['type'] === 'text/plain' || $file['type'] === 'application/vnd.ms-excel') {
        $handle = fopen($file['tmp_name'], 'r');
        if ($handle !== false) {
            $added = 0;
            $errors = 0;
            try {
                $pdo->beginTransaction();
                while (($data = fgetcsv($handle)) !== false) {
                    $code = strtoupper(trim($data[0]));
                    if (strlen($code) <= 10 && preg_match('/^[A-Z0-9]+$/', $code)) {
                        // Use INSERT IGNORE to prevent fatal errors on duplicates
                        $stmt = $pdo->prepare("INSERT IGNORE INTO access_codes (code) VALUES (?)"); 
                        $stmt->execute([$code]);
                        $added += $stmt->rowCount();
                    } else {
                        $errors++;
                    }
                }
                $pdo->commit();
                showMessage("Uploaded CSV: $added codes added, $errors invalid codes skipped.", $errors > 0);
            } catch (PDOException $e) {
                $pdo->rollBack();
                showMessage('Error uploading CSV: ' . $e->getMessage(), true);
            }
            fclose($handle);
        } else {
             showMessage('Error reading uploaded file.', true);
        }
    } else {
        showMessage('Invalid file type. Upload a CSV or text file.', true);
    }
    $redirect_needed = true;
}

// Handle code deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_code'])) {
    $code = trim($_POST['delete_code']);
    try {
        $stmt = $pdo->prepare("DELETE FROM access_codes WHERE code = ?");
        $stmt->execute([$code]);
        $stmt = $pdo->prepare("DELETE FROM active_sessions WHERE code = ?");
        $stmt->execute([$code]);
        showMessage('Code deleted successfully.', false);
    } catch (PDOException $e) {
        showMessage('Error deleting code: ' . $e->getMessage(), true);
    }
    $redirect_needed = true;
}

// Redirect after POST if needed (Post/Redirect/Get pattern)
if ($redirect_needed) {
    // Redirect to self to prevent form resubmission
    header('Location: admin.php');
    exit;
}

// -----------------------------------------------------
// 3. FETCH DATA FOR DISPLAY
// -----------------------------------------------------

// Get current video ID
$currentVideoId = '';
try {
    $stmt = $pdo->query("SELECT setting_value FROM settings WHERE setting_key = 'active_video_id'");
    $currentVideoId = $stmt->fetchColumn() ?: '';
} catch (PDOException $e) {
    error_log("DB Error fetching video ID: " . $e->getMessage());
    $currentVideoId = 'DB_ERROR'; // Placeholder for error state
}

// Hero settings
$heroTitle = '';
$heroSynopsis = '';
$heroImage = '';
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key IN ('hero_title','hero_synopsis','hero_image')");
    $stmt->execute();
    // The above returns multiple rows; fetch all and map
    $map = [];
    foreach ($stmt->fetchAll(PDO::FETCH_NUM) as $row) { /* no-op mapping here */ }
} catch (PDOException $e) {
    // fallback: fetch individually
}
try { $heroTitle = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_title'")->fetchColumn() ?: ''; } catch (PDOException $e) {}
try { $heroSynopsis = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_synopsis'")->fetchColumn() ?: ''; } catch (PDOException $e) {}
try { $heroImage = $pdo->query("SELECT setting_value FROM settings WHERE setting_key='hero_image'")->fetchColumn() ?: ''; } catch (PDOException $e) {}

// Admin Metrics: Current viewers and Avg saved position (Continue Watching)
$metrics = ['current_viewers' => 0, 'avg_position' => 0];
try {
    $ttl = getActiveSessionTTL();
    // current viewers = active sessions with last_activity within TTL
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM active_sessions WHERE last_activity > (NOW() - INTERVAL $ttl MINUTE)");
    $stmt->execute();
    $metrics['current_viewers'] = (int)$stmt->fetchColumn();

    if (!empty($currentVideoId) && $currentVideoId !== 'DB_ERROR') {
        $stmt = $pdo->prepare("SELECT AVG(position_seconds) FROM viewing_progress WHERE video_id = ?");
        $stmt->execute([$currentVideoId]);
        $avg = $stmt->fetchColumn();
        $metrics['avg_position'] = $avg !== null ? (int)$avg : 0;
    }
} catch (PDOException $e) {
    error_log('DB Error computing metrics: ' . $e->getMessage());
}

// Recent audit events
$recentAudits = [];
try {
    $stmt = $pdo->query("SELECT event_time, event_type, code, session_id FROM audit_logs ORDER BY id DESC LIMIT 20");
    $recentAudits = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) { error_log('DB audits: ' . $e->getMessage()); }

$videos = [];
try {
    $stmt = $pdo->query("SELECT id, youtube_id, title FROM videos ORDER BY id DESC");
    $videos = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) { error_log('DB videos: ' . $e->getMessage()); }

// Entitlements search + pagination
$entQ = isset($_GET['ent_q']) ? trim($_GET['ent_q']) : '';
$entPageSize = 10;
$entPage = isset($_GET['ent_page']) && ctype_digit($_GET['ent_page']) && (int)$_GET['ent_page'] > 0 ? (int)$_GET['ent_page'] : 1;
$entOffset = ($entPage - 1) * $entPageSize;
$entWhere = '';
$entParams = [];
if ($entQ !== '') {
    $entWhere = "WHERE ce.code LIKE ? OR v.title LIKE ?";
    $like = "%$entQ%";
    $entParams = [$like, $like];
}
$entTotal = 0; $entPages = 1; $entitlements = [];
try {
    // Count
    $sqlCount = "SELECT COUNT(*) FROM code_entitlements ce JOIN videos v ON v.id=ce.video_id $entWhere";
    $stmt = $pdo->prepare($sqlCount);
    $stmt->execute($entParams);
    $entTotal = (int)$stmt->fetchColumn();
    $entPages = max(1, (int)ceil($entTotal / $entPageSize));
    if ($entPage > $entPages) { $entPage = $entPages; $entOffset = ($entPage - 1) * $entPageSize; }
    // Page
    $sql = "SELECT ce.code, v.title, v.id as video_id FROM code_entitlements ce JOIN videos v ON v.id=ce.video_id $entWhere ORDER BY ce.code, v.title LIMIT $entPageSize OFFSET $entOffset";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($entParams);
    $entitlements = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) { error_log('DB entitlements: ' . $e->getMessage()); }

// Schedules search + pagination
$schQ = isset($_GET['sch_q']) ? trim($_GET['sch_q']) : '';
$schPageSize = 10;
$schPage = isset($_GET['sch_page']) && ctype_digit($_GET['sch_page']) && (int)$_GET['sch_page'] > 0 ? (int)$_GET['sch_page'] : 1;
$schOffset = ($schPage - 1) * $schPageSize;
$schWhere = '';
$schParams = [];
if ($schQ !== '') {
    $schWhere = "WHERE v.title LIKE ?";
    $schParams = ["%$schQ%"];
}
$schTotal = 0; $schPages = 1; $schedules = [];
try {
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM schedules s JOIN videos v ON v.id=s.video_id $schWhere");
    $stmt->execute($schParams);
    $schTotal = (int)$stmt->fetchColumn();
    $schPages = max(1, (int)ceil($schTotal / $schPageSize));
    if ($schPage > $schPages) { $schPage = $schPages; $schOffset = ($schPage - 1) * $schPageSize; }
    $stmt = $pdo->prepare("SELECT s.id, v.title, s.starts_at, s.ends_at FROM schedules s JOIN videos v ON v.id=s.video_id $schWhere ORDER BY s.starts_at DESC LIMIT $schPageSize OFFSET $schOffset");
    $stmt->execute($schParams);
    $schedules = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) { error_log('DB schedules: ' . $e->getMessage()); }

// Get all codes
$accessCodes = [];
try {
    $stmt = $pdo->query("SELECT ac.code, COUNT(asess.code) as active_count, MAX(asess.last_activity) as last_activity
                         FROM access_codes ac
                         LEFT JOIN active_sessions asess ON ac.code = asess.code
                         GROUP BY ac.code
                         ORDER BY ac.code ASC");
    $accessCodes = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("DB Error fetching codes: " . $e->getMessage());
    // Handle error gracefully on the display side
}

// -----------------------------------------------------
// 4. ADMIN PANEL HTML (DISPLAY)
// -----------------------------------------------------
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background-color: #111827; color: white; font-family: 'Inter', sans-serif; }
        .bg-gray-800 { background-color: #1f2937; }
        .border-purple-500\/50 { border-color: rgba(168, 85, 247, 0.5); }
        .text-purple-400 { color: #a78bfa; }
        .input-text { background-color: #1f2937; border: 1px solid #374151; color: white; padding: 8px; border-radius: 6px; }
        /* Style for the custom file upload button */
        .file-upload-wrapper { overflow: hidden; position: relative; display: inline-block; cursor: pointer; }
        .file-upload-wrapper input[type=file] { position: absolute; left: 0; top: 0; opacity: 0; cursor: pointer; }
    </style>
</head>
<body class="p-4 md:p-8">

    <div class="max-w-4xl mx-auto">
        <h1 class="text-3xl font-bold text-purple-400 mb-6 border-b pb-2">Video Access Admin Panel</h1>

        <?php $message = getMessage('message'); if ($message): ?>
            <div class="bg-green-600/30 text-green-200 p-3 rounded-lg mb-4"><?= htmlspecialchars($message) ?></div>
        <?php endif; ?>
        <?php $error = getMessage('error'); if ($error): ?>
            <div class="bg-red-600/30 text-red-200 p-3 rounded-lg mb-4"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>

        <div class="bg-gray-800 p-6 rounded-lg shadow-xl mb-8 border border-purple-500/50">
            <h2 class="text-xl font-semibold mb-4 border-b border-gray-700 pb-2">Live Metrics</h2>
            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div class="bg-gray-700/50 p-4 rounded-lg border border-gray-600">
                    <p class="text-gray-300 text-sm">Current Viewers (last <?= htmlspecialchars((string)$ttl) ?> min)</p>
                    <p class="text-2xl font-bold text-white mt-1"><?= htmlspecialchars((string)$metrics['current_viewers']) ?></p>
                </div>
                <div class="bg-gray-700/50 p-4 rounded-lg border border-gray-600">
                    <p class="text-gray-300 text-sm">Average Saved Position (seconds)</p>
                    <p class="text-2xl font-bold text-white mt-1"><?= htmlspecialchars((string)$metrics['avg_position']) ?></p>
                </div>
            </div>
        </div>

        <div class="bg-gray-800 p-6 rounded-lg shadow-xl mb-8 border border-purple-500/50">
            <h2 class="text-xl font-semibold mb-4 border-b border-gray-700 pb-2">Hero Settings</h2>
            <form method="POST" action="admin.php">
                <div class="mb-3">
                    <label class="block text-sm mb-1">Title</label>
                    <input type="text" name="hero_title" value="<?= htmlspecialchars($heroTitle) ?>" class="input-text w-full">
                </div>
                <div class="mb-3">
                    <label class="block text-sm mb-1">Synopsis</label>
                    <textarea name="hero_synopsis" class="input-text w-full" rows="3"><?= htmlspecialchars($heroSynopsis) ?></textarea>
                </div>
                <div class="mb-3">
                    <label class="block text-sm mb-1">Background Image URL</label>
                    <input type="url" name="hero_image" value="<?= htmlspecialchars($heroImage) ?>" class="input-text w-full" placeholder="https://...">
                </div>
                <button type="submit" name="save_hero" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition">
                    Save Hero
                </button>
            </form>
            <div class="mt-4">
                <h3 class="text-lg font-semibold mb-2">Current Entitlements (<?= count($entitlements) ?>)</h3>
                <form method="GET" action="admin.php" class="mb-3 flex gap-2 items-center">
                    <input type="text" name="ent_q" value="<?= htmlspecialchars($entQ) ?>" placeholder="Search by code or title" class="input-text">
                    <button type="submit" class="bg-gray-600 hover:bg-gray-500 text-white rounded px-3 py-1">Search</button>
                </form>
                <ul class="list-disc ml-6 text-gray-300">
                    <?php foreach ($entitlements as $e): ?>
                        <li class="flex items-center justify-between gap-3">
                            <span>
                                <span class="font-mono text-purple-300"><?= htmlspecialchars($e['code']) ?></span>
                                → <?= htmlspecialchars($e['title']) ?> (Video ID <?= (int)$e['video_id'] ?>)
                            </span>
                            <form method="POST" action="admin.php" onsubmit="return confirm('Remove entitlement for code <?= htmlspecialchars($e['code']) ?>?');">
                                <input type="hidden" name="del_ent_code" value="<?= htmlspecialchars($e['code']) ?>">
                                <input type="hidden" name="del_ent_video_id" value="<?= (int)$e['video_id'] ?>">
                                <button type="submit" name="delete_entitlement" class="text-red-400 hover:text-red-300 text-sm">Remove</button>
                            </form>
                        </li>
                    <?php endforeach; ?>
                </ul>
                <?php if ($entPages > 1): ?>
                <div class="mt-3 flex gap-2 items-center">
                    <?php if ($entPage > 1): $p=$entPage-1; ?>
                        <a class="px-2 py-1 bg-gray-700 rounded text-white" href="?ent_q=<?= urlencode($entQ) ?>&ent_page=<?= $p ?>">Prev</a>
                    <?php endif; ?>
                    <span class="text-gray-300 text-sm">Page <?= $entPage ?> of <?= $entPages ?></span>
                    <?php if ($entPage < $entPages): $n=$entPage+1; ?>
                        <a class="px-2 py-1 bg-gray-700 rounded text-white" href="?ent_q=<?= urlencode($entQ) ?>&ent_page=<?= $n ?>">Next</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="bg-gray-800 p-6 rounded-lg shadow-xl mb-8 border border-purple-500/50">
            <h2 class="text-xl font-semibold mb-4 border-b border-gray-700 pb-2">Recent Audit Events</h2>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-700">
                    <thead>
                        <tr class="text-xs uppercase tracking-wider text-gray-400">
                            <th class="px-6 py-3 text-left">Time</th>
                            <th class="px-6 py-3 text-left">Event</th>
                            <th class="px-6 py-3 text-left">Code</th>
                            <th class="px-6 py-3 text-left">Session</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700">
                        <?php if (empty($recentAudits)): ?>
                            <tr><td colspan="4" class="px-4 py-3 text-center text-gray-500">No events</td></tr>
                        <?php else: foreach ($recentAudits as $row): ?>
                            <tr>
                                <td class="px-6 py-2 text-sm text-gray-300"><?= htmlspecialchars($row['event_time']) ?></td>
                                <td class="px-6 py-2 text-sm text-purple-300"><?= htmlspecialchars($row['event_type']) ?></td>
                                <td class="px-6 py-2 text-sm text-gray-300 font-mono"><?= htmlspecialchars($row['code'] ?? '') ?></td>
                                <td class="px-6 py-2 text-xs text-gray-400 font-mono"><?= htmlspecialchars(substr($row['session_id'] ?? '', 0, 10)) ?>...</td>
                            </tr>
                        <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="bg-gray-800 p-6 rounded-lg shadow-xl mb-8 border border-purple-500/50">
            <h2 class="text-xl font-semibold mb-4 border-b border-gray-700 pb-2">Video Settings</h2>
            <form method="POST" action="admin.php">
                <div class="mb-4">
                    <label for="admin_video_url" class="block text-sm font-medium mb-2">Current YouTube Video ID or URL</label>
                    <input type="text" name="admin_video_url" id="admin_video_url" value="<?= htmlspecialchars($currentVideoId) ?>" class="input-text w-full" placeholder="Enter 11-char ID or full YouTube URL" required>
                </div>
                <button type="submit" name="save_video" class="bg-purple-600 hover:bg-purple-700 text-white font-bold py-2 px-4 rounded transition">
                    Update Video
                </button>
                <a href="/public/index.php" class="ml-4 text-purple-400 hover:text-purple-300">View Player</a>
            </form>
        </div>

        <div class="bg-gray-800 p-6 rounded-lg shadow-xl mb-8 border border-purple-500/50">
            <h2 class="text-xl font-semibold mb-4 border-b border-gray-700 pb-2">Access Code Management</h2>

            <form method="POST" action="admin.php" class="flex gap-4 mb-6 items-center">
                <input type="text" name="new_code" class="input-text flex-grow max-w-sm" placeholder="Add New Code (e.g., ABC123)" maxlength="10" required>
                <button type="submit" name="add_code" class="bg-green-600 hover:bg-green-700 text-white font-bold py-2 px-4 rounded transition">
                    Add Code
                </button>
            </form>

            <form method="POST" action="admin.php" enctype="multipart/form-data" class="flex gap-4 mb-6 items-center">
                <div class="file-upload-wrapper">
                    <button type="button" class="bg-gray-600 hover:bg-gray-700 text-white py-2 px-4 rounded transition">
                        Choose CSV File
                    </button>
                    <input type="file" name="csv_file" accept=".csv, text/plain" required>
                </div>
                <button type="submit" name="upload_csv" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition">
                    Upload Codes (CSV)
                </button>
                <span class="text-sm text-gray-400">CSV should contain one code per row.</span>
            </form>

            <h3 class="text-lg font-medium mt-6 mb-3 border-b border-gray-700 pb-1">Current Codes (<?= count($accessCodes) ?>)</h3>
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-700">
                    <thead>
                        <tr class="text-xs uppercase tracking-wider text-gray-400">
                            <th class="px-6 py-3 text-left">Code</th>
                            <th class="px-6 py-3 text-left">Active Sessions</th>
                            <th class="px-6 py-3 text-left">Last Activity</th>
                            <th class="px-6 py-3 text-right">Action</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700">
                        <?php if (empty($accessCodes)): ?>
                            <tr><td colspan="4" class="px-6 py-4 text-center text-gray-400">No access codes found.</td></tr>
                        <?php endif; ?>

                        <?php foreach ($accessCodes as $code): ?>
                            <tr class="hover:bg-gray-700">
                                <td class="px-6 py-4 font-mono text-sm"><?= htmlspecialchars($code['code']) ?></td>
                                <td class="px-6 py-4 text-sm">
                                    <span class="<?= $code['active_count'] > 0 ? 'text-red-400 font-bold' : 'text-green-400' ?>">
                                        <?= $code['active_count'] ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-400">
                                    <?= $code['last_activity'] ? htmlspecialchars($code['last_activity']) : 'N/A' ?>
                                </td>
                                <td class="px-6 py-4 text-right">
                                    <form method="POST" action="admin.php" onsubmit="return confirm('Are you sure you want to delete code <?= htmlspecialchars($code['code']) ?>?');">
                                        <input type="hidden" name="delete_code" value="<?= htmlspecialchars($code['code']) ?>">
                                        <button type="submit" class="text-red-500 hover:text-red-400 text-sm transition">
                                            Delete
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <a href="index.php?logout=true" class="text-red-500 hover:text-red-400 transition">
            <svg class="w-5 h-5 inline-block mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path></svg>
            Logout Admin
        </a>
    </div>

</body>
</html>